/** Header file for the Autometrics automatic model selection code. Use as
<xmp>
#import  <packages/autometrics/autometrics>
</xmp>
at the top of Ox files using it. This assumes installation in Ox/packages/autometrics.
**/

#ifndef INC_AUTOMETRICS
#define INC_AUTOMETRICS

/** This class is used as storage objects for terminal models found by Autometrics.
All data members are public.
The only methods are the constructor and `AutometricsStore::Copy`.
**/
class AutometricsStore
{
	AutometricsStore(...);
	Copy(const objSrc);		// Copy contents from AutometricsStore object
public:	
	/**
	selection indices in GUM of maintained components and regressors
	(1<tt>x</tt><i>k</i>; for the initial GUM this is:
	<tt>vec(m_oGUM.AutometricsComponentMap())' ~ 1 - vec(m_oGUM.AutometricsRegressorMap())'</tt>;
	for reduced models it is a subset of the keepmap of the GUM)
	*/
	decl m_vKeepMap;
	decl m_bIsOK;			///< used internally by Autometrics

	decl m_dLogLik;			///< log-likelihood (m_oGUM.GetLogLik())
	decl m_cKept;			///< no of terms kept in model (<i>k</i>)
	decl m_vDiagnostics;	///< p-values for diagnostic tests (&lt;&gt; or 1<tt>x</tt><i>d</i> from <tt>vec(m_oGUM.AutometricsDiagnostics(0))'</tt>)
	decl m_vKeepTests;		///< p-values for significance tests of components and regressors (&lt;&gt; or 1<tt>x</tt><i>k</i> from <tt>vec(m_oGUM.AutometricsSignificance())'</tt>)
	decl m_cParFree;		///< total no free of parameters in the model (<i>p</i>, from <tt>m_oGUM.GetFreeParCount()</tt>)
	decl m_vParFree;		///< estimated parameters for re-estimation	 (<i>p</i><tt>x</tt>1 from <tt>m_oGUM.GetFreePar()</tt>)
	decl m_mCovar;			///< Variance-covariance matrix of coefficients	(&lt;&gt; unless Wald tests are used or bounds specified, then <i>p</i><tt>x</tt><i>p</i> matrix from <tt>m_oGUM.AutometricsCovar()</tt>)
	decl m_mCovarMap;		///< map of free terms (&lt;&gt; unless Wald tests are used or bounds specified, then <i>k</i><tt>x</tt><i>p</i> 0,1 matrix from <tt>m_oGUM.AutometricsCovarMap()</tt>)
	decl m_vBoundTests;		///< significance of lower bounds (1: not-binding; &lt;&gt; if no bounds are specified)
	decl m_vKeepCoefs;		///< coefficients of components and regressors (or &lt;&gt;)
	decl m_vModelStats;		///< dditional model info for final report
}

class AUTO
{
public:
	enum
	{
		CUSTOM = -1,
		HUGE	,
		STANDARD,
		DEFAULT = STANDARD,
		MEDIUM	,
		SMALL	,
		TINY	,
		MINUTE	,
		USER
	}
	enum
	{
		TEST_LRF,
		TEST_LR,
		TEST_WF,
		TEST_W
	}
	enum
	{
		FINALGUM = 0,
		AIC,
		HQ,
		SC,
		MINSIZE,
		Cp,
		MAXLOGLIK,
		FIRSTMODEL,
		TIE_TOP
	}
}

/** The Autometrics settings class.
Get/Set functionality is provided to change any value,
but all data members are also public.
**/
class AutometricsSettings
{
	AutometricsSettings();
	~AutometricsSettings();

	Set(const oAutoSet);
	Default();
	ReportSettings();
	static ScalePValue(const dPValue, const dFactor);

	SetStrategyPValue(const dPValue);
	SetStrategySize(const iLevel);
	SetChopLags(const bChopLags, ...);
	SetChopVars(const bChopVars);
	SetOutliers(const bAddOutliers, ...);
	SetBacktestingGUM(const iBacktestingGUM);
	SetEffort(const iEffort);
	SetTiebreaker(const iTiebreaker);
	SetPruning(const iPrune);
	SetPValueBounds(const dPVal);
	SetBlocking(const dBlockAtFraction);
	SetBlockEffort(const iBlockEffort);
	GetBlockEffort();
	SetBlockMaxSize(const mxBlockSize);
	SetBlockBacktesting(const iBlockBacktesting);
	SetPrint(const iLevel);
	SetRunOnce(const bSet);
	GetPrint();
	IsFixedStrategy();
	GetStrategySize();
	GetPValueDel_Prune();
	GetPValuePrune();
	GetPValueInit();
	GetPValueDiagnostics();
	GetPValueBounds();
	GetRunOnce();
	GetBacktestingGUM();
	GetPruning();
	GetEffort();
	GetTiebreaker();
	GetOutliers();
	GetPValueOutliers();
	GetPValueBacktest();
	SetPValueBacktest(const dPVal);
	SetPValuePrune(const dPValDel, const dPValBunch, const dPValChop, const dPValSqueeze);
	SetPValueInit(const dPValCut1_Init, const dPValChop_Init);
	SetPValueDiagnostics(const dPVal);
	SetTestMode(const iSet);
	SetMaxTerminals(const iSet);
	SetMaxEstimates(const iSet);
	GetMaxTerminals();
	GetMaxEstimates();
	SetRemoveFailedBounds(const bSet);
	GetRemoveFailedBounds();

public:
	// autometrics settings
	decl m_cMaxEst;				///< upperbound for no of estimated models (-1: unlimited)
	decl m_cMaxTerm;			///< upperbound for no of terminal candidate models (-1: unlimited)
	decl m_iEffort;				///< effort of search
	decl m_dPValDiagnostics;	///< cut-off critical value for diagnostic tests

	decl m_dPValSqueeze_Prune;	///< cut-off p-value for single regressor significance test (LR)
	decl m_dPValDel_Prune;		///< cut-off p-value for regressor significance test (LR) against current GUM
	decl m_dPValBunch_Prune;	///< p-value for bunching
	decl m_dPValChop_Prune;		///< p-value for chopping
	decl m_dPValBacktest;		///< p-value for backtesting
	decl m_dPValBounds;			///< p-value for bounds

	decl m_dPValCut1_Init;		///< minimal p-value for initial chopping significance test
	decl m_dPValChop_Init;		///< cut-off p-value for regressor significance test (LR) against current GUM
	decl m_dPValOutliers;		///< p-value (2-sided) for outlier detection

	decl m_bTestChiSq;			///< TRUE: tests are Chi^2, else use F(-approx)
	decl m_bTestWald;			///< TRUE: tests use covariance (Wald), else loglik (LR)
	decl m_sTest;				///< test label
	decl m_iTestMode;			///< Test type: AUTO_TEST_...
	decl m_iTiebreaker;			///< 0:Union 1:AIC 2:HQ 3:SC
	decl m_iPrintLevel;			///< determines amount of output
	decl m_iBacktestingGUM;		///< >0: test reduction against GUM (2: strict)
	decl m_iPruning;			///< 1: do pruning, 2: pruning and chopping (default)
	decl m_bChopLags;			///< TRUE: reduce the lag length
	decl m_iChopLagMode;		///< 1: keep the lag form closed
	decl m_bChopVars;			///< TRUE: reduce variables
	decl m_bAddOutliers;		///< TRUE: use outlier detection
	decl m_iRunOnce;			///< TRUE: stop after first encompassing pass

	decl m_bRemoveFailedBounds;	///< TRUE: remove variables that cause a bound violation

	decl m_dBlockAtFraction;	///< ratio of k/T to start using blocking
	decl m_iBlockEffort;		///< 0-5 block-search method (1=default)
	decl m_mxBlockSize;			///< <=0: no maximum, >0: maximum block size
	decl m_iBlockBacktesting;	///< 0: not at all, 1: only in expansion, 2: in expansion and reduction
}

/** Implements the Autometrics algorithm, for internal use.
**/
class AutometricsWork : AutometricsSettings
{
	AutometricsWork();
	~AutometricsWork();
	GetVersion();
	Clear();

protected:	
	static NextSubset(const cN, vSubsetPrev);

	P_Value_LogLik(const dLogLik, const dLogLikRes, const cP, const cPRes);
	P_Value_RSS(const mRSS, const mRSSRes, const cP, const cPRes);
	P_Value_Covar(const vP, const mCovar, const mCovarMap, const vKeepMap, const vKeepMapRes);
	P_Value_Node(const aModel, const vKeepMapRes);
	P_Value_Root(const vKeepMap, const aModelRes);
	P_Value_Models(const aModel, const aModelRes);
	GetCovarMap(const aModel, const vKeepMapRes);
	
	GetcY();
	GetDiagnosticsGUM(const iGUM);
	GetEstimationCount();
	SetLowerBounds(const mR, const vR, const asR, const vRIsExact);
	SetActiveDiagnostics(const bActiveDiagnostics);
	UpdatePValDiag(const vDiagnostics);
	UpdateRIsExact(const vBounds);
	Initialize(const oGUM, const iYvar);
	AddOutlierDummies(const iXvar);
	static GetRegNames(const oGUM);
	static GetFixedRegNames(const oGUM, const iYvar);
	GetRegressorsFromMap(const vMap);
	GetComponentsFromMap(const vMap);
	GetRemovedBounds();
	Sort(const vRegTests, const bTopDown, const dPVal1);
	GetBunch(const aModel, const vCandidateMap, const iInitialBunchSize, const vExclusionMap,
		const dPValDel, const dPValBunch, const bGrowPValBunch, const dPValBacktest,
		const bDoDiagnostics, const aModels, const cModels, const sMsg);
	IsSemiTerminal(const vFreeSet, const oModel);
	RefineIncompleteTerminal(k, const dPValBunch, const bGrowPValBunch,
		const bDoDiagnostics, const aModels, const aFinalModels);
	CheckTerminal(k, const aModels, const iGumIndex);
	Prune(aFinalModels, const oStartModel, const aModelsPrev, const dPValBunch,
		const dPValSqueeze, const mxTerm, const mxNestedBranches, const flSettings,
		const sMsg);
	GetMaxLag(const oModel);
	InitChopLags(const iChopLagMode, const dPVal1, const bShrinkPVal1, const dPValDel,
		const dPVal1Min, const iYvarExcluded, const bSetNewGum);
	InitChopVars(const dPVal1, const bShrinkPVal1, const dPValDel, const dPVal1Min, const bSetNewGum);
	GetKeepMapFromSelection(const oModel);
	Encompassing(const iNewGum, const bTest, const dPValDel, const mxPasses);
	ClearEncompassedTerminals(const oModel, const dPValDel, const sEncompassing);
	UpdateUnionFromTerminals(const aoModel, const bUpdate);
	ClearFailedTerminals();
	ClearDominatedTerminals();
	Report(const cRegGum, const cEst, const cTermCandidates);
	CheckInsignificance(const iModel, const bReport);
	ClearUnfinishedTerminals();
	static InfoCrit(const dLogLik, const cP, const cT);
	RunTiebreaker();
	PrintTerminals(const iModSel, const sGum, const sTerminal);
	PrintTerminalCoefficients(const iModSel, const sGum, const sTerminal);

	ResetInitialGum();
	FormulateReduced(const vDropMap);
/*	static FormulateFromSelArray(const oGUM, const aSel);*/
	EstimateReduced(const vDropMap);
	EstimateReducedFromKeepMap(const vKeepMap);
	StoreReduced(const aDest, const oGUM, const vKeepMap, const vDiagnostics, const vBounds, const bUpdateAll);
	EvaluateBounds(const vKeepMapRes, const vPRes, const mCovarRes);
	CountBoundFailures(const vPValBounds);
	CountDiagnosticFailures(const vPValDiag);
	EvaluateReduced(const aDest, const vDropMap, const aModels, const cModels,
		const dPValDel, const dPValBacktest, const bDoDiagnostics, const bPrint);
	HasIdenticalTerminal(const aFinalModels, const vKeepMap);
	HasNestedTerminal(const aFinalModels, const vKeepMap);
	HasEncompassingTerminal(const aFinalModels, const oNewModel, const bReplace, const iPrintLevel);
	GetNestedTerminalContrast(const aFinalModels, const iFirst, const vKeepMap, const vFreeMap);
	GetNestedUnionContrast(const aFinalModels, const iFirst, const vKeepMap, const vFreeMap);
	PrintTerminal(const sLbl, const iIndex, const oModel);
	AddTerminal(aFinalModels, const oModel, const bCheck, const iPrintLevel);
	UpdateGUM(const aFinalModels, const oNewModel);
	Reduce(const oGUM, const iYvar, const iXvar, const iUvar, ...);
	static RunStepwise(const oGUM, const iYvar, const iXvar, const iUvar, const iLasso, const dPValue, const maxK, const iCrit, const bPrintTrace);
	static OrderStepwise(const oGUM, const iYvar, const iXvar, const iUvar, const bLasso, dPValue, cBlockSize, const bPrintTrace);
	ReduceStepwise(const oGUM, const iYvar, const iXvar, const iUvar, const iLasso, const maxK, const iCrit);
	ReduceSize(const oGUM, const iYvar, const iXvar, const iUvar, const bDummy, const cK, const iCrit);
	
	BlockRun(const oGUM, const iPass, const cK, const iYvar, const iXvar, const iUvar, const asel_base,
		const asel_gum, const regmap_gum, const idx_fixed, iBlockType, bDoCrossBlocks,
		bDelayDiagnostics, const iPrintLevel);

	static BlockOrderStepwise(const oGUM, const bOrder, const bLasso, const dPValue, const cK,
		const iYvar, const iXvar, const iUvar, const asel_base, const aselGUM,
		const regmap_free, const regmap_fixed, const iPrintLevel);
	static BlockOmitted(const auto, const oGUM, dPVal_omitted, const iFactor, const dPVal_Bounds,
		const mxSel, const cK, const iYvar, const iXvar, const iUvar, const asel_base,
		const asel_gum, const regmap_free, const regmap_fixed, const iBlockType,
		const bCanGrow, const bDoCrossBlocks, bDelayDiagnostics, const iPrintLevel, const sPrefix);
	static BlockReduceRegs(const oGUM, dSampleFraction, const iYvar, const iXvar,
		const iUvar, const iKeepYuptoLag, const iKeepXuptoLag, const sKeepVar, const dPValue,
		const iReductionEffort, const iExtensionEffort, iBlockEffort, const mxBlockSize,
		const iBlockBacktesting, const iBacktestingGUM, const bRemoveFailedBounds, const iPrintLevel);

protected:
	static decl sm_sVersion;
	
	// initial GUM, with respect to which all models are estimated
	decl m_gum;					///< initial GUM estimates, holds master map
	decl m_gum0;				///< GUM 0, same as initial GUM if no presearches
	decl m_aGumSelInfo;			///< initial GUM model formulation
	decl m_aGumSelInfoType;		///< var types for initial GUM model formulation
	decl m_asNames;				///< component~regressor names
	decl m_cY;					///< no of equations
	decl m_cT;					///< estimation sample size

	// optional bounds for the GUM
	decl m_bHaveBounds;			///< TRUE: have bound constraints
	decl m_mR, m_vR, m_asR;		///< specification of bounds
	decl m_vRIsExact;			///< 0:inexact bounds >0:exact bounds
	
	// variables that are changed as the search progresses
	decl m_aFinalModels;		///< [0] is active GUM, rest are terminal candidates
	decl m_oGUM;				///< Modelbase object, constantly modified
	decl m_cEst;				///< no of estimations done sofar
	decl m_vPValDiagAct;		///< actual critical values for each diagnostic
	decl m_asDiagnostics;		///< names of the diagnostic tests
	decl m_vRIsExactAct;		///< updated for the GUM: 0=inexact >0=exact <0=inactive
	decl m_vRIsRemoved;			///< updated for the GUM: 1:violated, variables removed
	decl m_bActiveDiagnostics;	///< TRUE: always test (only used for comparison in MC)
}

/** The Autometrics class.
**/
class Autometrics : AutometricsWork
{
	Autometrics();
	~Autometrics();

	GetFinalModel(const k);
	EstimateFinalModel(const k);
	static MALag(const oDb, const sX, const mxLag, const iPower);
	static DummySaturation(const oGUM, const iXvar, const sType);
	static ReadDateField(sX, const aiYear1, const aiPeriod1);
	static IsImpulseDummy(const sX, const aiYear1, const aiPeriod1, const aiOp, const aiYear2, const aiPeriod2);
	static IsTypeDummy(const sType, const sX, const aiYear1, const aiPeriod1);
	static IsDummy(const sX);
	static DbRenewDummies(const oDb, asX);
	static GetDummy(const oDb, const sX);
	Estimate(const oGUM, const iYvar, const iXvar, ...);
	EstimateLasso(const oGUM, const iYvar, const iXvar, const iUvar, const maxK, const iCrit);
	EstimateStepwise(const oGUM, const iYvar, const iXvar, const iUvar, const maxK, const iCrit);
	EstimateBackward(const oGUM, const iYvar, const iXvar, const iUvar, const maxK, const iCrit);
	EstimateSize(const oGUM, const iYvar, const iXvar, const iUvar, const maxK, const iCrit);
	EncompassModel(const iFinalModel1, const oModel2);
}	
#endif // INC_AUTOMETRICS
