#include <oxstd.h>

#import "simstore"
#import "simdesigns"

#ifdef USE_PCGIVE
  #import <packages/PcGive/pcgive_ects>
#else
  #define USE_PCFIML
  #import "./pcfimlex"
#endif

#import "./autometrics"

#include "sim_autometrics.oxh"

/**
Creates a Monte Carlo experiment and selects a design.
@param sDGP name of DGP<br>
	case "HP":		`HP`<br>
	case "HPBig":	`HPBig`<br>
	case "JEDC":	`JEDC`<br>
	case "JEDC_EJ":	`JEDC_EJ`<br>
	case "D1":		`Breaks`<br>
	case "D2":		`Breaks`<br>
	case "D3":		`Breaks`<br>
	case "Bc":		`Bc`<br>
	case "B20":		`B20`<br>
	case "Bct":		`Bct`<br>
	case "MBc":		`MBc`<br>
	case "MBct":	`MBct`<br>
	case "BLc":		`BLc`<br>
	case "IUc":		`IUc`<br>
	case "BUc":		`BUc`<br>
	case "MIUc":	`MIUc`<br>
	case "MBUc":	`MBUc`<br>
	case "BLcx":	`BLcx`<br>
	case "BUcx":	`BUcx`<br>
	case "BUcx_t":	`BUcx_t`<br>
@param aDGPno number of DGP	variation, or array with DGP options
*/
AutometricsExp::AutometricsExp(const sDGP, const aDGPno)
{
	BaseExp();
	m_oAutoSet = new AutometricsSettings();
	m_oAutoSet.SetPrint(0);

	m_iAutoOutliers = 0;

	decl model;
#ifdef USE_PCFIML
	model = new PcFimlEx();
	model.SetTestLag(-1);	// -1: use default lag lengths for AR and ARCH tests
	model.SetChowFrac(0.7);	// PcGive Default
#endif
#ifdef USE_PCGIVE
	model = new PcGive();
#endif
	model.SetPrint(FALSE);	

	// this will hand ownership of model over to the design
	SelectDesign(model, sDGP, aDGPno, Y_VAR, X_VAR, U_VAR);
}
AutometricsExp::~AutometricsExp()
{
	~BaseExp();
}
/**
Main Autometrics settings.
@param dPValue 	significance for model selection (1 for no selection)
@param sOutliers one of: "none","large","iis","sis","iis+sis","diis","diis+iis"
@param bChopLags TRUE: lag presearch; FALSE: no	lag presearch
*/
AutometricsExp::Autometrics(const dPValue, const sOutliers, const bChopLags)
{
	BaseExp::Autometrics(dPValue, sOutliers, bChopLags);
	
	m_oAutoSet.Default();
	m_oAutoSet.SetPrint(0);
	m_oAutoSet.SetStrategyPValue(dPValue);

	switch_single(strlwr(sOutliers))
	{
		case "large":	m_iAutoOutliers = 1;
		case "dummy":	m_iAutoOutliers = 2;
		case "impulse":	m_iAutoOutliers = 2;
		case "iis":		m_iAutoOutliers = 2;
		case "sis":		m_iAutoOutliers = 3;
		case "iis+sis":	m_iAutoOutliers = 4;
		case "sis+iis":	m_iAutoOutliers = 4;
		case "diis":	m_iAutoOutliers = 5;
		case "diis+iis":m_iAutoOutliers = 6;
		case "iis+diis":m_iAutoOutliers = 6;
		case "iis+sis+tis":	m_iAutoOutliers = 7;
		default:		m_iAutoOutliers = 0;
	}
	
	m_bChopLags = m_oAutoSet.m_bChopLags = bChopLags != 0;
}
/**
Set Autometrics option.
@param sOpt 	option label
@param val option value
*/
AutometricsExp::AutometricsSet(const sOpt, const val)
{
	switch_single(strlwr(sOpt))
	{
		case "pre-lag":				m_oAutoSet.SetChopLags(val);
		case "pre-var":				m_oAutoSet.SetChopVars(val);
		case "effort":				m_oAutoSet.SetEffort(val);
		case "backtesting":
			switch_single(strlwr(val))
			{
				case "none": 			m_oAutoSet.SetBacktestingGUM(0);
				case "gum0": 			m_oAutoSet.SetBacktestingGUM(1);
				case "current gum" : 	m_oAutoSet.SetBacktestingGUM(2);
			}
		case "tie-breaker":
			switch_single(strlwr(val))
			{
				case "union": 	m_oAutoSet.SetTiebreaker(AUTO::FINALGUM);
				case "aic": 	m_oAutoSet.SetTiebreaker(AUTO::AIC);
				case "hq" : 	m_oAutoSet.SetTiebreaker(AUTO::HQ);
				case "sc" : 	m_oAutoSet.SetTiebreaker(AUTO::SC);
				case "min": 	m_oAutoSet.SetTiebreaker(AUTO::MINSIZE);
				case "cp" : 	m_oAutoSet.SetTiebreaker(AUTO::Cp);
				case "max": 	m_oAutoSet.SetTiebreaker(AUTO::MAXLOGLIK);
			}
		case "print":				m_oAutoSet.SetPrint(val);
		case "pvalue_tests":		m_oAutoSet.SetPValueDiagnostics(val);		
		case "block_fraction":		m_oAutoSet.SetBlocking(val);		
		case "block_method":		m_oAutoSet.SetBlockEffort(val);
		case "block_max":			m_oAutoSet.SetBlockMaxSize(val);
#ifdef USE_PCFIML
		case "test_default":		
		case "test_hetero":			m_oDesign.GetRefModelObject().SetHeteroTest(val);
		case "test_normality":		m_oDesign.GetRefModelObject().SetHeteroTest(val);
#endif
#ifdef USE_PCGIVE
		case "test_default":		m_oDesign.GetRefModelObject().AutometricsSet("test_default", val);
		case "test_hetero":			m_oDesign.GetRefModelObject().AutometricsSet("test_hetero", val);
		case "test_normality":		m_oDesign.GetRefModelObject().AutometricsSet("test_normality", val);
#endif
	}
}

/**
Run Stepwise.
*/
AutometricsExp::Stepwise(const sMethod, const dPValue, const maxK, const iCrit)
{
	BaseExp::Stepwise(sMethod, dPValue, maxK, iCrit);

	m_oAutoSet.SetStrategyPValue(dPValue);	 // still needed
}
/**
Run the experiment.
@param cM 	no of replications
@param iReportSettings 	report Autometrics Settings
@param sSaveFirstRepName 	optional filename to save first replication of data set (using Database::Save), default is "" for no saving
*/
AutometricsExp::Run(const cM, const iReportSettings, const sSaveFirstRepName)
{
	// reset the seed: all experiments run off the same seed!
	ranseed(-1);

	decl refmodel = m_oDesign.GetRefModelObject(), asdiag = {};

	CreateStore(cM);
	m_oDesign.PreGenerate(refmodel);

	decl i, time = timer(), final;
	for (i = 0; i < cM; ++i)
	{
		decl model = clone(refmodel);
		decl auto = new Autometrics();
		auto.Set(m_oAutoSet);
		
		if (model.GetPrint() > 0)
			println("\n============== experiment ", i+1);

		m_oDesign.Generate(model, i);

		if (i == 0 && isstring(sSaveFirstRepName) && sSaveFirstRepName != "")
			model.Save(sSaveFirstRepName);
		m_oDesign.FormulateModel(model);

		switch_single (m_sMethod)
		{
			case "Autometrics":
				if (m_dAutoPval < 1)
					final = auto.Estimate(model, Y_VAR, X_VAR, U_VAR);
				else
				{
					model.Estimate();
					final = -1;
				}
			case "Stepwise":
				final = auto.EstimateStepwise(model, Y_VAR, X_VAR, U_VAR, m_maxK, m_iStopCrit);
			case "Lasso":
				final = auto.EstimateLasso(model, Y_VAR, X_VAR, U_VAR, m_maxK, m_iStopCrit);
			case "Backward":
				final = auto.EstimateBackward(model, Y_VAR, X_VAR, U_VAR, m_maxK, m_iStopCrit);
			case "Auto_size":
				final = auto.EstimateSize(model, Y_VAR, X_VAR, U_VAR, m_maxK, 0);
			default:
				oxrunerror("Unknown method");
		}
		m_oDesign.UndoFormulateModelDeltaY(model);	// in case Dy instead of Y
		
		StoreModel(model, SimStore::STORE_REP, i);
		if (final >= 0)
			m_oStore.StoreDiagnostics(SimStore::STORE_GUM, i, auto.GetDiagnosticsGUM(0));

		m_oDesign.FormulateDGP(model);
		model.Estimate();

		StoreModel(model, SimStore::STORE_DGP, i);
		if (final >= 0)
			m_oStore.StoreEncompassing(i, auto.EncompassModel(final, model));

		if (i == 0)		// PcGive: can only get names after estimation
			asdiag = model.AutometricsDiagnosticNames();
			
		delete model;
	    delete auto;
	}

	decl ret_val;
	serial		  // serial to keep output together, in case called in parallel loop
	{
		println("\n====================");
		if (iReportSettings)
		{
			if (m_sMethod == "Autometrics")
			{
				decl auto = new Autometrics();	   // needed for report
				auto.Set(m_oAutoSet);
	
				println("Model selection: ", "%12s", m_dAutoPval < 1 ? m_sMethod : "NONE");
				println("replications, M=", "%13d", cM,     "  time", "%25s", timespan(time));
				if (m_sMethod == "Autometrics" && m_dAutoPval < 1)
				{
					println("Autometrics settings");
					auto.ReportSettings();
					println("Autometrics version ", "%9s", auto.GetVersion(),  "  block effort        ", "%9d", auto.GetBlockEffort());
				}
			    delete auto;
			}
			else
			{
				println("Model selection: ", "%12s", m_sMethod, "  termination: ", "%16s",
					m_iStopCrit > 0 ? {"","AIC","HQ","SC","","C_p"}[m_iStopCrit] :
						m_iStopCrit == -1 ? sprint("size=", m_maxK) : sprint("pvalue=", m_dAutoPval));
				println("replications, M=", "%13d", cM,     "  time", "%25s", timespan(time));
			}
		}
	
		ret_val = m_oStore.ReportSelection(m_oDesign.GetTitle(), m_oDesign.GetRefUParCount(),
			m_oDesign.GetObsCount(), m_dAutoPval);
		m_oStore.ReportDiagnostics(asdiag);
		println("GUM for diagnostic testing is GUM0\n");
	
		if (iReportSettings == 2)
			m_oStore.ReportPar({}, {});
    }
	return ret_val;
}
