#include <oxstd.h>
#include <arma.h>

#include "simdesign.oxh"

///////////////////////////////////////////////////////////////////////
/**
Constructor: sets client object and variable type identifiers.
SimDesign takes a Modelbase-derived object to formulate the reference model into.
The DGP and model are then (sub)sets of this reference model.
SimDesign then creates data into a clone of the reference model.
@param obj reference to Modelbase derived client object. NB	SimDesign takes ownership
of this object, and will delete it when the destructor is called.
@param iYvar identifies endogenous variables
@param iXvar identifies free regressors
@param iUvar identifies unrerstricted (fixed) regressors
**/
SimDesign::SimDesign(const obj, const iYvar, const iXvar, const iUvar)
{
	m_oRefModel = obj;
	m_iYvar = iYvar;
	m_iXvar = iXvar;
	m_iUvar = iUvar;
	m_sTitle = "";
	m_iDGP = -1;
	m_asY = m_asX = m_asEps = m_asStep = {};
	
	m_bRenewX = FALSE;
	m_bHaveX = FALSE;
	m_bModelDeltaY = FALSE;
	
	m_asPar_RM = {};
	m_vParDGP_RM = m_vParModelNonZero_RM = <>;
	m_iConstant_RM = USE_CONSTANT_NONE;
	m_bTrend_RM = FALSE;
	m_cStep_RM = 0;
	m_cTdiscard = m_cTest = 0;
	m_asDummy = {};

	m_dSigmaDGP = 1;
	m_dARParDGP = 0;
	m_dXcorrelation = 0;
	m_iXcorrelationMode = USE_XCOR_MATCONST;
}
SimDesign::~SimDesign()
{
	delete m_oRefModel;
	m_oRefModel = 0;
}
/**
Sets the title
**/
SimDesign::SetTitle(const sTitle)
{
	m_sTitle = sTitle;
}
/**
Gets the title
**/
SimDesign::GetTitle()
{
	return m_sTitle;
}
/**
Gets the client object reference
**/
SimDesign::GetRefModelObject()
{
	return m_oRefModel;
}
/**
Sets the DGP number. This can be used by derived class to denote variations of the DGP.
**/
SimDesign::SetDGPno(const iDGP)
{
	m_iDGP = iDGP;
}
/**
Gets the DGP number.
**/
SimDesign::GetDGPno()
{
	return m_iDGP;
}
/**
Creates database for the DGP in the client object
@param sFile if string: file to load
@param cTmax no of observations in the created database
@param cXmax no of regressors k in the created database
@param vSteps empty or Sx2 matrix with step indices. This creates i=0,..,S-1 steps, wich are one
for observations vSteps[i][0] : vSteps[i][1], and zero otherwise.
<p>
First the database is created (from file or empty) with cTmax observations.
Deterministic(2) is called to add Constant, Trend and Seasonal.
Then y, Dy and u are appended, followed by x1,..,xk. Finally, if vSteps is not <>
then it specifies the step dummies to add to the DGP, with names Step[#:#] where # is the
observation index.
<p>
Remember that an Ox program starts with the default seed. I normally use the
convention to run each experiment of the default seed (so related experiments use
the same data). However, this could cause problems if random data is created here,
because it can be identical to that in the first replication.
**/
SimDesign::CreateDatabase(const sFile, const cTmax, const cXmax, const vSteps)
{
	decl i;

	if (isstring(sFile) && sFile != "")
	{
		m_oRefModel.Load(sFile);
		decl ct = m_oRefModel.GetSize();
		if (cTmax > 0 && ct < cTmax)
			m_oRefModel.Grow(cTmax - ct);
		m_bHaveX = TRUE;
	}
	else
	{
		m_oRefModel.Database::Create(1, 1, 1, 1, 1);
		m_oRefModel.Grow(cTmax - 1);
		m_bHaveX = FALSE;
	}	
    m_oRefModel.Deterministic(2);

	m_asY = { "y" };
	m_asEps = { "u" };

	m_oRefModel.Append(constant(.NaN, cTmax, 1), m_asY);
	m_oRefModel.Append(constant(.NaN, cTmax, 1), "D" ~ m_asY[0]);
	m_oRefModel.Append(constant(.NaN, cTmax, 1), m_asEps);
	
	m_asX = new array[cXmax];
	for (i = 0; i < cXmax; ++i)
	{
		m_asX[i] = sprint("x", i + 1);
		m_oRefModel.Append(constant(.NaN, cTmax, 1), m_asX[i]);
	}

	decl t1, t2, v, csteps = sizer(vSteps);
	if (csteps && sizec(vSteps) != 2)
		oxrunerror("Need cSteps x 2 matrix");
	m_asStep = new array[csteps];
	for (i = 0; i < csteps; ++i)
	{
		t1 = max(int(vSteps[i][0]), 0);
		t2 = int(vSteps[i][1]);
		t2 = t2 >= 0 ? min(t2, cTmax - 1) : cTmax - 1;
		v = zeros(cTmax, 1);
		v[t1 : t2] = 1;
		m_asStep[i] = sprint("Step[", t1, ":", t2, "]");
		m_oRefModel.Append(v, m_asStep[i]);
	}
}
/**
Creates the design, which is the reference model for the DGP and GUM
@param cYlag no of lags for dependent variable
@param cX    no of regressors
@param cXlag  lag length for regressors
@param iConstantMode  treatment of Constant: USE_CONSTANT_NONE, USE_CONSTANT_U, USE_CONSTANT_X   
@param bTrend TRUE for a Trend in the GUM
@param cStep  number of steps in the GUM: added to the database by `SimDesign::CreateDatabase`, and selected in that order.
@param cTdiscard no of observations to discard. The actual number discarded is max(cTdiscard,cYlag,cXlag).
@param cTest no of observations to use for estimation
<p>
Calls `SimDesign::FormulateReferenceModel` to formulate the reference model in the client object.
Calls `SimDesign::SetModel` to make the model the same as the reference model (this can be changed later).
**/
SimDesign::CreateReference(const cYlag, const cX, const cXlag, const iConstantMode,
	const bTrend, const cStep, const cTdiscard, const cTest)
{
	m_cYlag_RM = cYlag;
	m_cX_RM = cX;
	m_cXlag_RM = cXlag;
	m_iConstant_RM = iConstantMode;
	m_cU_RM = iConstantMode == USE_CONSTANT_U ? 1 : 0;
	m_bTrend_RM = bTrend;
	m_cStep_RM = cStep;
	m_cTdiscard = cTdiscard;

	if (m_cTdiscard	&& m_oRefModel.GetVar("Trend")[0] == 1)
		m_oRefModel.Renew(m_oRefModel.GetVar("Trend") - m_cTdiscard, "Trend");
	
	decl it1 = max(cYlag, cXlag, cTdiscard), it2, ct = m_oRefModel.GetSize();
	m_iYear1est = m_oRefModel.ObsYear(it1);
	m_iPeriod1est = m_oRefModel.ObsPeriod(it1);
	it2 = cTest > 0 ? it1 + cTest - 1 : ct - 1;
	m_iYear2est = m_oRefModel.ObsYear(it2);
	m_iPeriod2est = m_oRefModel.ObsPeriod(it2);
	m_cTest = it2 - it1 + 1;
	// formulate the reference model in the client object
	m_asPar_RM = FormulateReferenceModel(m_oRefModel);

	// set the model to be the same as the reference model (can be changed later)
	SetModel(ones(sizeof(m_asPar_RM), 1));
}
/**
Sets the DGP parameters in terms of the reference model.
@param vPar parameters for the DGP, corresponding to the order in the reference model.
**/
SimDesign::SetDGP(const vPar)
{
	m_vParDGP_RM = vec(vPar);
}
/**
Sets the DGP error properties.
@param dESE standard error (default 1)
@param dARPar autoregressive parameter (default 0)
**/
SimDesign::SetDGPError(const dESE, const dARPar)
{
	m_dSigmaDGP = dESE;
	m_dARParDGP = dARPar;
}
/**
Creates a correlation matrix for X by setting rho. 
@param dCorrelation rho
@param bConstant USE_XCOR_MATCONST: one on diagonal, rho otherwise; USE_XCOR_MATPOWIJ: rho^|i-j|;
USE_XCOR_AUTO: autocorrelated regressors, scaled by sqrt(1 - rho^2)
NB With USE_XCOR_AUTO the first observation will be zero. Ideaaly this is taken into
account by incrementing no of discards when calling `SimDesign::CreateReference`.
**/
SimDesign::SetXCorrelation(const dCorrelation, const iMode)
{
	m_dXcorrelation = dCorrelation;
	m_iXcorrelationMode = iMode;
}
/**
Sets the DGP parameters in terms of the reference model.
@param vPar parameters for the DGP, corresponding to the order in the reference model.
A non-zero value indicates that the variable is in the model.
**/
SimDesign::SetModel(const vParNonZero)
{
	m_vParModelNonZero_RM = vec(vParNonZero);
}
/**
Removes variables by name from the model.
@param asPar array of names to remove (give zero parameter in reference model).
**/
SimDesign::RemoveFromModel(const asPar)
{
	// find the index/indices of these parameters
	decl sel = deletec(matrix(strfind(m_asPar_RM, asPar)), -1);
	// and set them to zero to exclude them from the Model used for the GUM
	m_vParModelNonZero_RM[sel] = 0;
}
/**
Set TRUE if data on X is to be renewed for every replication, FALSE otherwise.
**/
SimDesign::SetRenewX(const bRenewX)
{
	m_bRenewX = bRenewX;
}
/**
Override this virtual function to add additional terms to the reference model
@returns array with added names ({} for none)
**/
SimDesign::AddExtrasToReference()
{
	return {};
}
/**
Saturates the client database with the specified dummies.
This should be called after `SimDesign::CreateDatabase` and before `SimDesign::CreateReference`.
@param sType type: "I:", "DI:", "S1:", "S".
@param cTdiscard no of leading dummy terms to discard.
**/
SimDesign::DummySaturation(const sType, const cTdiscard)
{
	decl cout = m_oRefModel.GetSize() - cTdiscard, mx = zeros(m_oRefModel.GetSize(), cout);
	decl it1 = cTdiscard, it2 = it1 + cout;
	decl ic1 = 0, ic2 = cout - 1;
	decl c = m_oRefModel.GetVarCount();
	
	switch (sType)				// create all the dummy variables
	{
		case "I:":				// impulse dummies, NB at index [0] is I:1 etc. e.g. I:3 is 0,0,1,0,0...
			mx[it1 : ][] = unit(cout);
			break;
		case "DI:":				// differenced impulse dummies
			mx[it1 : ][] = diff0(unit(cout), 1);
			ic1 = 1;
			break;
		case "S1:":				// step dummies (one then zero), e.g. S1:3 is 1,1,1,0,0... so zero from index [3] onwards
			mx[it1 : ][] = setupper(unit(cout), 1);
			if (it1)
				mx[ : it1 - 1][] = 1;
			++ic1;				// drop the impulse dummy for the first obs
			--ic2;				// drop the intercept
			break;
		case "S:":				// step dummies (zero then one)
			mx[it1 : ][] = setupper(unit(cout), 1);
			if (it1)
				mx[ : it1 - 1][] = 1;
			mx = lag0(1 - mx, -1, 1);
			++ic1;				// drop the impulse dummy for the first obs
			--ic2;				// drop the intercept
			break;
		default:
			oxrunerror(sType ~ " unknown saturation type");
			return 0;
	}

	// create the labels
	decl as = new array[ic2 - ic1 + 1];
	for (decl i = 0; i < sizeof(as); ++i)
	{
		as[i] = sType ~ m_oRefModel.GetSampleByIndex(ic1 + it1 + i, -1);
	}
	// update the database
	m_oRefModel.Append(mx[][ic1 : ic2], as);
	
	// assume they're not in the database yet
	m_asDummy ~= as;

	return cout - ic1;
}
/**
Adds a break to a parameter vector formulated in terms of impulse dummies that are in the client database.
This should be called after `SimDesign::CreateReference`.
@param vRefPar reference parameter vector to add break to
@param dFraction
>=2	break of size dSize in every dFraction'th observation
>=0	break of size dSize in first round(dFraction * m_cTest) observations
>=-1 break of size dSize in last round(-dFraction * m_cTest) observations
>=-100 break of size dSize in 5 blocks of round(-dFraction/100 * m_cTest / 5) obs
evenly spread as: [bbbb .... bbbb .... bbbb .... bbbb .... bbbb]
@param dSize scalar specifying size for each break (== 0: do nothing), or vector of break sizes
(one entry for each break, first entry is for first break value, etc.).
@returns vRefPar modified for break
**/
SimDesign::AddImpulseBreakToRefPar(vRefPar, const dFraction, const dSize, const bUseSteps)
{
	if (dFraction == 0 || dSize == 0)
		return vRefPar;

	if (sizerc(vRefPar) != sizeof(m_asPar_RM))
	{
		oxrunerror("vRefPar dimension does not match reference model");
	}
	
	decl vp = zeros(m_cTest, 1);
	
	if (dFraction >= 2)
	{
		// break of size dSize in every dFraction'th observation
		for (decl j = 0; j < m_cTest; j += dFraction)
		{
			vp[j] = sizerc(dSize) == 1 ? dSize : dSize[j];
		}
	}
	else if (dFraction >= 0)
	{
		// break of size dSize in first round(dFraction * m_cTest) observations
		for (decl j = 0, dt = round(dFraction * m_cTest); j < dt; ++j)
		{
			vp[j] = sizerc(dSize) == 1 ? dSize : dSize[j];
		}
	}
	else if (dFraction >= -1)
	{
		// break of size dSize in last round(-dFraction * m_cTest) observations
		for (decl j = m_cTest + round(dFraction * m_cTest), k = 0; j < m_cTest; ++j, ++k)
		{
			vp[j] = sizerc(dSize) == 1 ? dSize : dSize[k];
		}
	}
	else if (dFraction >= -100)
	{
		// break of size dSize in 5 blocks of round(-dFraction/100 * m_cTest / 5) obs
		// evenly spread as: [bbbb .... bbbb .... bbbb .... bbbb .... bbbb]
		decl dt = round(-dFraction / 100 * m_cTest);
		decl ct5 = int(dt / 5), sel = zeros(1, m_cTest), it;

		vp[ : ct5 - 1] = sizerc(dSize) == 1 ? dSize : dSize[0 : ct5 - 1];				// start
		it = int(m_cTest / 4 - ct5 / 2);	 											// 2nd block
		vp[it : it + ct5 - 1] = sizerc(dSize) == 1 ? dSize : dSize[ct5 : 2 * ct5 - 1];
		it = int(m_cTest / 2 - ct5 / 2);	 											// middle
		vp[it : it + ct5 - 1] = sizerc(dSize) == 1 ? dSize : dSize[2 * ct5 : 3 * ct5 - 1];			 
		it = int(3 * m_cTest / 4 - ct5 / 2); 											// third block
		vp[it : it + ct5 - 1] = sizerc(dSize) == 1 ? dSize : dSize[3 * ct5 : 4 * ct5 - 1];
		vp[m_cTest - ct5 : ] = sizerc(dSize) == 1 ? dSize : dSize[4 * ct5 : 5 * ct5 - 1];// end
 	}
	// now try to find the impulse dummies in the database
	decl sdum1, sdum2, ddum1, ddum2, idx, cb, k, t1 = m_oRefModel.GetIndex(m_iYear1est, m_iPeriod1est);

	for (decl j = 0; j < m_cTest; ++j)
	{
		if (vp[j] == 0)
			continue;

		sdum1 = sdum2 = ddum1 = ddum2 = 0;
		// determine length of break
		cb = 1;
		if (bUseSteps)
			for (k = j + 1; k < m_cTest; ++k, ++cb)
			{
				if (vp[k] != vp[k - 1])
					break;
			}
		if (cb > 1 && bUseSteps)
		{
			if (j > 0)
			{
				sdum1 = "S1:" ~ m_oRefModel.GetSampleByIndex(t1 + j - 1, -1);
				ddum1 = -vp[j];
			}
			if (k <	m_cTest)
			{
				sdum2 = "S1:" ~ m_oRefModel.GetSampleByIndex(t1 + k, -1);
				ddum2 = vp[j];
			}
			j = k - 1;
		}
		else
		{
			sdum1 = 0;
			sdum2 = "I:" ~ m_oRefModel.GetSampleByIndex(t1 + j, -1);
			ddum2 = vp[j];
		}
		if (isstring(sdum1))
		{
			idx = strfind(m_asPar_RM, sdum1);
			if (idx == -1)
				oxrunerror(sdum1 ~ " not found in reference model");
			vRefPar[idx] = ddum1;
		}
		if (isstring(sdum2))
		{
			idx = strfind(m_asPar_RM, sdum2);
			if (idx == -1)
				oxrunerror(sdum2 ~ " not found in reference model");
			vRefPar[idx] = ddum2;
		}
	}
	
	return vRefPar;
}
///////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////
/**
@returns residual standard deviation of the DGP (default is 1)
**/
SimDesign::GetSigmaInDGP()
{
	return m_dSigmaDGP;
}
/**
@returns residual autoregressive coefficient of the DGP (default is 0)
**/
SimDesign::GetARParInDGP()
{
	return m_dARParDGP;
}
/**
@returns no of parameters in the DGP
**/
SimDesign::GetDgpParCount()
{
	return int(sumc(m_vParDGP_RM .!= 0));
}
/**
@returns no of U variables in the reference model
**/
SimDesign::GetRefUParCount()
{
	return m_cU_RM;
}
/**
@returns no of observations for estimation
**/
SimDesign::GetObsCount()
{
	return m_cTest;
}
/**
@returns the DGP parameters in terms of the reference model
**/
SimDesign::GetRefParDGP()
{
	return m_vParDGP_RM;
}
/**
@returns the names of the parameters in the reference model
**/
SimDesign::GetRefParNames()
{
	return m_asPar_RM;
}
/**
@returns no of observations that are discarded
**/
SimDesign::GetObsInitCount()
{
	return max(m_cYlag_RM, m_cXlag_RM, m_cTdiscard);
}
/**
@returns the label of the constant mode
**/
SimDesign::GetRefConstantLabel()
{
	decl asconstant = {"USE_CONSTANT_NONE", "USE_CONSTANT_U", "USE_CONSTANT_X"};
	return asconstant[m_iConstant_RM];
}
/**
Set TRUE to use Dy as dependent variable in FormulateModel, FALSE otherwise.
**/
SimDesign::UseModelDeltaY(const bUseDiff)
{
	m_bModelDeltaY = bUseDiff;	
}
/**
Returns TRUE if using Dy as dependent variable in FormulateModel, FALSE otherwise (default).
**/
SimDesign::IsModelDeltaY()
{
	return m_bModelDeltaY;
}
///////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////
/**
Formulates the reference model in the client object.
@param oClient target client object
@returns array with parameter names
The order of variables is:
1) y
2) lags of y: y_1,..,y_p as Yvar
3) X regressors: x1,..x1_r, x2,..x2_r,  ...
4) Step breaks created in `SimDesign::CreateDatabase` as Xvar
5) Dummies as Xvar
6) `SimDesign::AddExtrasToReference`
7) Trend (if any) as Xvar
8) Constant (none, Xvar or Uvar)
So the constant is always the last term.
**/
SimDesign::FormulateReferenceModel(const oClient)
{
	decl aspar, asx;
	oClient.DeSelect();
	oClient.ClearModel();

	oClient.Select(m_iYvar, { m_asY[0], 0, m_cYlag_RM } );
	
	for (decl i = 0; i < m_cX_RM; ++i)
	{
		oClient.Select(m_iXvar, { m_asX[i], 0, m_cXlag_RM } );
	}

	for (decl i = 0; i < m_cStep_RM; ++i)
	{
		oClient.Select(m_iXvar, { m_asStep[i], 0, 0 } );
	}

	decl t1 = oClient.GetIndex(m_iYear1est, m_iPeriod1est);
	decl t2 = oClient.GetIndex(m_iYear2est, m_iPeriod2est);

	for (decl i = 0; i < sizeof(m_asDummy); ++i)
	{
		if (varc(oClient.GetVar(m_asDummy[i])[t1:t2][]) > 1e-10)
		{
			oClient.Select(m_iXvar, { m_asDummy[i], 0, 0 } );
		}
	}
    oClient.GetGroupLagNames(m_iYvar, 1, 1000, &aspar);
    oClient.GetGroupNames(m_iXvar, &asx);
	aspar ~= asx;

	aspar ~= AddExtrasToReference();
	
	if (m_bTrend_RM)
	{
		oClient.Select(m_iXvar, { "Trend", 0, 0 } );
		aspar ~= "Trend";
	}
	if (m_iConstant_RM == USE_CONSTANT_U)
	{
		oClient.Select(m_iUvar, { "Constant", 0, 0 } );
		aspar ~= "Constant";
	}
	else if (m_iConstant_RM == USE_CONSTANT_X)
	{
		oClient.Select(m_iXvar, { "Constant", 0, 0 } );
		aspar ~= "Constant";
	}

	// cannot use SetSelSample because y and X data missing at this stage
	oClient.ForceSelSample(m_iYear1est, m_iPeriod1est, m_iYear2est, m_iPeriod2est);

	return aspar;
}
/**
Formulates a model in the client object with respect to the reference model.
@param oClient target client object
@param vParInRef parameter values, non-zero selects the corresponding term
@param iConstantMode mode for the Constant (if included)
@param bUseDeltaY if TRUE, the model is formulated with Dy instead of y
The order of variables is:
1) y or Dy
1) Constant (if any: Xvar or Uvar)
2) Trend (if any) as Xvar
2) lags of y: y_1,..,y_p as Yvar
3) X regressors: x1,..x1_r, x2,..x2_r,  ...
4) Step breaks created in `SimDesign::CreateDatabase` as Xvar
5) Dummies as Xvar
6) `SimDesign::AddExtrasToReference`
So the constant is always the last term.
**/
SimDesign::FormulateModelInReference(const oClient, const vParInRef, const iConstantMode, const bUseDeltaY)
{
	decl i, j, c = sizerc(vParInRef), svar, lag;

	if (sizeof(m_asPar_RM) != c)
		oxrunerror("Parameter dimension error");
		
	oClient.DeSelect();
	oClient.ClearModel();

	if (bUseDeltaY)
		oClient.Select(m_iYvar, { "D" ~ m_asY[0], 0, 0 });
	else
		oClient.Select(m_iYvar, { m_asY[0], 0, 0 });
	
	for (i = 0; i < sizerc(vParInRef); ++i)
	{
		if (vParInRef[i] != 0)
		{
			svar = m_asPar_RM[i];

			if (svar == "Constant")
			{
				oClient.Select(iConstantMode == USE_CONSTANT_U ? m_iUvar : m_iXvar, { "Constant", 0, 0 } );
				continue;
			}			
			if (svar == "Trend")
			{
				oClient.Select(m_iXvar, { "Trend", 0, 0 } );
				continue;
			}			
			lag = 0;
			j = strfind(svar, '_');
			if (j > 0)
			{
				if (sscan(svar[j + 1:], "%d", &lag) != 1)
					oxrunerror("lag expected in name " ~ svar);
				svar = svar[ : j - 1];
			}
			oClient.Select(svar == m_asY[0] ? m_iYvar : m_iXvar, { svar, lag, lag });
		}
	}
    oClient.SetSelSample(m_iYear1est, m_iPeriod1est, m_iYear2est, m_iPeriod2est);
}
/**
Formulates the DGP model in the client object.
@param oClient the client object. 
**/
SimDesign::FormulateDGP(const oClient)
{
	FormulateModelInReference(oClient, m_vParDGP_RM, m_iConstant_RM, FALSE);
}
/**
Formulates the model in the client object.
@param oClient the client object. 
**/
SimDesign::FormulateModel(const oClient)
{
	FormulateModelInReference(oClient, m_vParModelNonZero_RM, m_iConstant_RM, m_bModelDeltaY);
}
/**
Reformulates the model in the client object from Dy to y, and re-estimates.
@param oClient the client object. 
**/
SimDesign::UndoFormulateModelDeltaY(const oClient)
{
	if (!m_bModelDeltaY)
		return;
		
	decl asel = oClient.GetSelInfo();
	decl db_idx_y = oClient.GetVarIndex(m_asY[0]);
	decl db_idx_dy = oClient.GetVarIndex("D" ~ m_asY[0]);
	
	decl idx_dy = vecindex(asel[0] .== db_idx_dy .&& asel[2] .== 0);
	decl idx_y1 = vecindex(asel[0] .== db_idx_y .&& asel[2] .== 1);

	// replace DY with Y as depvar
	if (idx_dy == <>)
		return;
	asel[0][idx_dy] = db_idx_y;

	oClient.DeSelect();
	oClient.ClearModel();
	oClient.SetSelInfo(asel);
	
	// add Y_1 to model if not there yet
	if (idx_y1 == <>)
		oClient.Select(m_iYvar, {m_asY[0], 1, 1});
		
	// re-estimate
    oClient.SetSelSample(m_iYear1est, m_iPeriod1est, m_iYear2est, m_iPeriod2est);
	oClient.Estimate();
}
///////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////
/**
X transformation immediately after drawing the N[0,1] terms.
@param mX generated X
@returns transformed X (default is same as input X)
**/
SimDesign::TransformX1(const mX)
{
	return mX;
}
/**
X transformation after applying correlation (if any) to X.
@param mX correlated X
@returns transformed X (default is same as input X)
**/
SimDesign::TransformX2(const mX)
{
	return mX;
}
/**
Generates X.
@param iRep replication index (0 is first)
@param cT sample size
@returns generated X
**/
SimDesign::GenerateX(const iRep, const cT)
{
	decl x = TransformX1(rann(cT, m_cX_RM));

	if (m_dXcorrelation)
	{
		if (m_iXcorrelationMode == USE_XCOR_AUTO)
		{	// first observation will be zero, ideaaly this is taken into
			// account by user incrementeing no of discards
			x = armagen(zeros(cT, m_cX_RM), sqrt(1 - m_dXcorrelation^2) * x, m_dXcorrelation, 1, 0);
		}
		else
		{
			decl chol;
			if (m_iXcorrelationMode == USE_XCOR_MATCONST)
			{
				chol = choleski(setdiagonal(constant(m_dXcorrelation, m_cX_RM, m_cX_RM), 1));
			}
			else if (m_iXcorrelationMode == USE_XCOR_MATPOWIJ)
			{
				chol = choleski(m_dXcorrelation .^ fabs(range(0, m_cX_RM-1)' - range(0, m_cX_RM-1)));
			}
			chol ./= chol[m_cX_RM - 1][m_cX_RM - 1];
			x *= chol';
		}
	}
 	return TransformX2(x);
}
/**
To be called after setting the initial seed (optional) for the experiment, but
before starting the Monte Carlo loop.
**/
SimDesign::PreGenerate(const oClient)
{
}
/**
Generates X, u, y, Dy, stores them in the client database.
@param oClient the object to generate data into. This should be a clone of the reference object (so
that the reference model is already formulated).
@param iRep replication index (0 is first)
**/
SimDesign::Generate(const oClient, const iRep)
{
	decl y, x = <>, u, ct = oClient.GetSize(), i;

	u = rann(ct, 1);
	if (m_dARParDGP)
		u = armagen(zeros(ct, 1), u, m_dARParDGP, 1, 0);
	oClient.Renew(u, m_asEps);

	if ((iRep == 0 && !m_bHaveX) || m_bRenewX)	// need to create the X variables
	{
		if (m_cX_RM)
		{
			x = GenerateX(iRep, ct);
			oClient.Renew(x, m_asX[ : m_cX_RM - 1]);
		}
		m_bHaveX = TRUE;
	}
	
	decl asel = oClient.GetSelInfo();
	decl idx_y = vecindex(asel[1] .== m_iYvar .&& asel[2] .== 0);
	// the X or U status of a variable doesn't matter for modelforc, but the Y status does
	y = modelforc(m_dSigmaDGP * u, oClient.GetAll(), oClient.GetVarIndex(m_asY[0]),
    	dropc(asel[0], idx_y), dropc(asel[2], idx_y), m_vParDGP_RM', 0);
		
	oClient.Renew(y, m_asY[0]);
	oClient.Renew(diff0(y, 1, .NaN), "D" ~ m_asY[0]);
}
///////////////////////////////////////////////////////////////////////

