#include <oxstd.h>
#include <arma.h>

#import "simdesign"
#import "simstore"
#include "simdesigns.oxh"

///////////////////////////////////////////////////////////////////////
/**
Hoover-Perez Monte Carlo experiments.
@param iDGP index of HP experiment: 1,..,9,17,80,800,8000,80000
*/
HP::HP(const iDGP, const obj, const iYvar, const iXvar, const iUvar)
{
	SimDesign(obj, iYvar, iXvar, iUvar);
    LoadData();
	
	if (iDGP >= 1)
		SetGUMandDGP(iDGP, USE_CONSTANT_U);
}
HP::DiffDb()
{
	decl i, dx;
	// 19 variables in database, last one is y
	for (i = 0; i <= 18; ++i)
	{
		// NB. set missing values created by differencing to 0.
		// This allows the use of modelforc without adjustments; just have
		// to make sure that these observations are not used in estimation
		if (i == 1 || i == 2 || i == 4 || i == 8 || i == 9 || i == 17)
			dx = diff0(GetRefModelObject().GetVarByIndex(i), 2, 0);
		else
			dx = diff0(GetRefModelObject().GetVarByIndex(i), 1, 0);
		GetRefModelObject().Renew(dx, sprint("x", i + 1));
	}
}
HP::LoadData()
{
    CreateDatabase("../data/HooverPerez(1999).xls", 0, 19, <>);
	DiffDb();
}
HP::GetParametersForDGP(const iDGP, const asParDesign)
{
	decl vpar = zeros(sizeof(asParDesign), 1), dese;

	decl iy_1  = strfind(asParDesign, "y_1");
	decl iy_2  = strfind(asParDesign, "y_2");
	decl ix3   = strfind(asParDesign, "x3");
	decl ix3_1 = strfind(asParDesign, "x3_1");
	decl ix11  = strfind(asParDesign, "x11");
	decl ix11_1= strfind(asParDesign, "x11_1");
	
	// NB: Table 3 has an error: u* needs sqrt(7/16) to get unit variance.
	//   u*_t = a u*_{t-1} + b u_t
	//   V[u*] = b^2 / (1-a^2)
	//   V[u*] = 1 --> b^2 = (1-a^2) = 1-9/16 = 7/16
	// Model 7' also has an error: the factor on u_t should be sqrt(7/16)9.73=6.44	(not 6.73)
	// Model 8' also has an error: the factor on x21_t should be 0.75*0.046=0.0345
	switch_single(iDGP)
	{
	case 1:
		dese = 130;
	case 2:
		vpar[iy_1] = 0.75;
		dese = sqrt(7/16) * 130;
	case 3:
		vpar[iy_1] = 0.395;
		vpar[iy_2] = 0.3995;
		dese = 0.00172;
	case 4:
		vpar[ix11] = 1.33;
		dese = 9.73;
	case 5:
		vpar[ix3] = -0.046;
		dese = 0.11;
	case 6:
		vpar[ix11] = 0.67;
		vpar[ix3] = -0.023;
		dese = 4.92;
	case 7:
		vpar[iy_1] = 0.75;
		vpar[ix11] = 1.33;
		vpar[ix11_1] = -0.9975;
		dese = sqrt(7/16) * 9.73;
	case 8:
		vpar[iy_1] = 0.75;
		vpar[ix3] = -0.046;
		vpar[ix3_1] = 0.0345;
		dese = 0.073;				// 0.11*sqrt(7)/4
	case 9:
		vpar[iy_1] = 0.75;
		vpar[ix3] =-0.023;
		vpar[ix3_1] = 0.01725;
		vpar[ix11] = 0.67;
		vpar[ix11_1] = -0.5025;
		dese = 3.25;
	case 17:						// 17 indicates modification of 7
		vpar[iy_1] = 0.75;
		vpar[ix11] = 1.33;
		vpar[ix11_1] = -0.9975;
		dese = sqrt(7/16) * 6.73;	// mistake as seen in PcGets simulations
	case 80:
		vpar[iy_1] = 0.75; vpar[ix3] = -0.046; vpar[ix3_1] = 0.0345; dese = 0.073 * 50;
	case 800:
		vpar[iy_1] = 0.75; vpar[ix3] = -0.046; vpar[ix3_1] = 0.0345; dese = 0.073 * 10;
	case 8000:
		vpar[iy_1] = 0.75; vpar[ix3] = -0.046; vpar[ix3_1] = 0.0345; dese = 0.073 / 10;
	case 80000:
		vpar[iy_1] = 0.75; vpar[ix3] = -0.046; vpar[ix3_1] = 0.0345; dese = 0.073 / 50;
	default:
		println("*** Unknown DGP ***");
	}

	return {vpar, dese};
}
HP::SetGUMandDGP(const iDGP, const iConstantMode)
{
	m_iDGP = iDGP;
	SetTitle(sprint("HP", iDGP, " constant=", {"None","U","X"}[iConstantMode]));

	CreateReference(4, 18, 1, iConstantMode, FALSE, 0, 6, 0);

	decl dese, vpar;
	[vpar, dese] = GetParametersForDGP(m_iDGP, m_asPar_RM);
	SetDGP(vpar);
	SetDGPError(dese, 0);
}
///////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////
// HPBig : HP
/**
Big Hoover-Perez Monte Carlo experiments.
@param iDGP index of HP experiment: 1,..,9,17,80,800,8000,80000
@param cExtra number of extra regressors
*/
HPBig::HPBig(const iDGP, const obj, const iYvar, const iXvar, const iUvar, const cExtra)
{
	m_cExtra = cExtra;
	HP(iDGP, obj, iYvar, iXvar, iUvar);
}
HPBig::LoadData()
{
    CreateDatabase("../data/HooverPerez(1999).xls", 0, 19 + m_cExtra, <>);
	DiffDb();
}
HPBig::SetGUMandDGP(const iDGP, const iConstantMode)
{
	m_iDGP = iDGP;
	SetTitle(sprint("HP", iDGP, "big constant=", {"None","U","X"}[iConstantMode]));

	CreateReference(4, 18 + m_cExtra, 4, iConstantMode, FALSE, 0, 6, 0);

	decl dese, vpar;
	[vpar, dese] = GetParametersForDGP(m_iDGP, m_asPar_RM);
	SetDGP(vpar);
	SetDGPError(dese, 0);

}
HPBig::PreGenerate(const oClient)
{
	// create here, because they are not created in the simulations
	for (decl i = 0; i < m_cExtra; ++i)
	{
		oClient.Renew(rann(oClient.GetSize(), 1), sprint("x", 19 + i + 1));
	}
}

///////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////
// JEDC_raw
JEDC_raw::JEDC_raw(const obj, const iYvar, const iXvar, const iUvar)
{
	SimDesign(obj, iYvar, iXvar, iUvar);
	m_iDummySaturation = SAT_NONE;
}
JEDC_raw::GetSaturationLabel()
{
	return {"None","IIS","SIS","IIS+SIS","DIS","DIS+IIS"}[m_iDummySaturation];
}
JEDC_raw::SetGUMandDGP(const iDGP, const cYlag, const cXtot, const cXlag, const cXsignificant,
	const iConstantMode, const cTest, const dRhoZ, const bDGPisTvalue,
	const iDummySaturation, const dFraction, const dSize, const dTrend, const dYlag)
{
	m_iDGP = iDGP;
	decl ctdiscard = (cYlag && dYlag == 1) ? 100 : max(cYlag, cXlag);
	decl have_ylag = cYlag && dYlag != 0;

	// ideally, when using USE_XCOR_AUTO ideally discard another observation, because
	// otherwise the first in the database will be zero

	decl dpar_constant = 0, dpar_trend = 0, btrendingum = FALSE;
	if (sizerc(dTrend) == 2)
	{
		dpar_constant = dTrend[0];
		if (dTrend[1] != 0)
			dpar_trend = dTrend[1];
	}
	else
	{
		dpar_trend = dTrend;
	}
	// add the trend to the GUM if it is in the DGP or if the coefficient is .NaN
	if (isnan(dpar_trend))
	{
		btrendingum = TRUE;
		dpar_trend = 0;
	}
	else
	{
		btrendingum = dpar_trend != 0;
	}
	m_iDummySaturation = iDummySaturation;

	// no need to discard initial observations when the lagged variables do not enter the DGP
	CreateDatabase("", ctdiscard + cTest, cXtot, <>);
	
	decl sylag = have_ylag ? sprint(",", dYlag, "*y_1") : "";
	SetTitle(sprint("JEDC_raw(DGP=", iDGP, ",k=", cXtot, ",",
		"n=", cYlag + cXtot * (cXlag + 1) + (iConstantMode == USE_CONSTANT_X) + btrendingum, ",",
		"n0=", cXsignificant, ",",
		"rho=", dRhoZ, bDGPisTvalue ? ",t=DGP" : ",t=8,6,4,3,2", ","
		"constant=", dpar_constant, "/T^0.5,trend=", dpar_trend, "/T^0.5",
		btrendingum ? " (trend in GUM)" : "", sylag, ", discard=", GetRefModelObject().GetSize() - cTest, ")\n"));
	SetTitle(GetTitle() ~ sprint("-", GetSaturationLabel(), "(", dFraction, ",", dSize, ")"));

	switch_single (iDummySaturation)
	{	case SAT_IIS:  		DummySaturation("I:", ctdiscard);
	 	case SAT_SIS:  		DummySaturation("S1:", ctdiscard);
	 	case SAT_IIS_SIS:  	DummySaturation("I:", ctdiscard);  DummySaturation("S1:", ctdiscard);
//	 	case SAT_DIS:  		DummySaturation("D:", ctdiscard);
//	 	case SAT_DIS_IIS:  	DummySaturation("D:", ctdiscard);  DummySaturation("I:", ctdiscard);
	}
	CreateReference(cYlag, cXtot, cXlag, iConstantMode, btrendingum, 0, ctdiscard, cTest == 0 ? 100 : cTest);

	if (iDGP == 199)
	{
		SetXCorrelation(dRhoZ, USE_XCOR_MATPOWIJ);
		SetTitle(GetTitle() ~ " C_X=rho^|i-j|");
	}
	else if (iDGP == 299)
	{
		SetXCorrelation(dRhoZ, USE_XCOR_MATCONST);
		SetTitle(GetTitle() ~ " C_X=rho");
	}
	else
		SetXCorrelation(dRhoZ, USE_XCOR_AUTO);
	SetRenewX(TRUE);
	
	decl vpar = zeros(sizeof(m_asPar_RM), 1);

	if (bDGPisTvalue)
	{
		for (decl i = 0; i < cXsignificant; ++i)
			vpar[strfind(m_asPar_RM, sprint("x", i+1))] = iDGP / sqrt(GetObsCount());
	}
	else
	{
		if (cXsignificant != 5)
			oxrunerror("Need 5 significant regressors");
		vpar[strfind(m_asPar_RM, "x1")] = 8 / sqrt(GetObsCount());
		vpar[strfind(m_asPar_RM, "x2")] = 6 / sqrt(GetObsCount());
		vpar[strfind(m_asPar_RM, "x3")] = 4 / sqrt(GetObsCount());
		vpar[strfind(m_asPar_RM, "x4")] = 3 / sqrt(GetObsCount());
		vpar[strfind(m_asPar_RM, "x5")] = 2 / sqrt(GetObsCount());
	}
	if (have_ylag)
		vpar[strfind(m_asPar_RM, "y_1")] = dYlag;
		
	if (dpar_constant != 0)
		vpar[strfind(m_asPar_RM, "Constant")] = dpar_constant / sqrt(GetObsCount());
	if (dpar_trend != 0)
		vpar[strfind(m_asPar_RM, "Trend")] = dpar_trend / sqrt(GetObsCount());
	
	if (dFraction != 0)
		vpar = AddImpulseBreakToRefPar(vpar, dFraction, dSize,
			iDummySaturation == SAT_SIS || iDummySaturation == SAT_IIS_SIS);
	
	SetDGP(vpar);
}
//JEDC_raw::SubstituteStepDummies(const asPar, const vPar, const vParSe, const vParPval, const iT1, const iT2)
//{
//	decl s, sdum, k, i, it, idx, asp = asPar, vp = vPar, vpse = vParSe, vpv = vParPval;
//	decl delset = <>;
//
//	// vParPval is shorter if there are unrestricted variables
//	if (sizer(vpv) < sizer(vp))
//		vpv |= zeros(sizer(vp) - sizer(vpv), 1);
//	
//	foreach (s in asPar[k])
//	{
//		if (sizeof(s) >= 4 && s[:2] == "S1:")
//		{
//println("found: ", s);
//			if (sscan(s[3:], "%d", &it) != 1)
//				continue;
//println("--it2= ", it, " ", iT1, " ", iT2);
//			for (i = it; i <= iT2 + 1; ++i)
//			{
//				sdum = sprint("I:", i);
//				idx = strfind(asp, sdum);
//println("--", sdum, " at ", idx);
//				if (idx == -1)
//					asp ~= sdum, vp |= vPar[k], vpse |= vParSe[k], vpv |= vParPval[k];
//				else
//					vp[idx] += vPar[k], vpse[idx] += vParSe[k], vpv[idx] += vParPval[k];
//			}
//			delset ~= k;
//		}
//	}
//	return {dropr(asp, delset), dropr(vp, delset), dropr(vpse, delset), dropr(vpv, delset)};
//}
///////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////
// JEDC : JEDC_raw
/**
JEDC Monte Carlo experiments.
@param dRhoZ correlation rho for X: C_X=rho^|i-j|
*/
JEDC::JEDC(const obj, const iYvar, const iXvar, const iUvar, const dRhoZ)
{
	JEDC_raw(obj, iYvar, iXvar, iUvar);

	SetGUMandDGP(dRhoZ == 0 ? 0 : 199, 1, 10, 1, 5, USE_CONSTANT_U, 100, dRhoZ, FALSE,
		0, 0, 0, 0, 0);
	SetTitle(sprint("JEDC(rho=", dRhoZ, ")\n"));
}
/**
EJ version of JEDC Monte Carlo experiments.
@param dRhoZ correlation rho for autocorrelated X's
*/
JEDC_EJ::JEDC_EJ(const obj, const iYvar, const iXvar, const iUvar, const dRhoZ)
{
	JEDC_raw(obj, iYvar, iXvar, iUvar);

	SetGUMandDGP(0, 1, 10, 1, 5, USE_CONSTANT_U, 100, dRhoZ, FALSE,
		0, 0, 0, 0, 0);
	SetTitle(sprint("JEDC_EJ(rho=", dRhoZ, ")\n"));
}
// JEDCdyn : JEDC_raw
/**
Dynamic version of JEDC Monte Carlo experiments: lagged X.
@param dRhoZ correlation rho for autocorrelated X's
*/
JEDCdyn::JEDCdyn(const obj, const iYvar, const iXvar, const iUvar)
{
	JEDC_raw(obj, iYvar, iXvar, iUvar);
}
JEDCdyn::SetGUMandDGP(const iDGP, const cYlag, const cXtot, const cXlag, const cXsignificant,
	const iConstantMode, const cTest, const dRhoZ, const bDGPisTvalue,
	const iDummySaturation, const dFraction, const dSize, const dTrend, const dYlag)
{
	JEDC_raw::SetGUMandDGP(iDGP, cYlag, cXtot, cXlag, cXsignificant, iConstantMode, cTest,
		dRhoZ, bDGPisTvalue, iDummySaturation, dFraction, dSize, dTrend, dYlag);

	SetTitle("Dyn" ~ GetTitle());
	decl vpar = zeros(sizeof(m_asPar_RM), 1);

	decl scale = sqrt(GetObsCount() * (1 - dRhoZ^2));
		
	if (bDGPisTvalue)
	{
		for (decl i = 0; i < cXsignificant; ++i)
		{
			vpar[strfind(m_asPar_RM, sprint("x", i+1))] = iDGP / scale;
			vpar[strfind(m_asPar_RM, sprint("x", i+1, "_1"))] = -iDGP / scale;
		}
	}
	else
	{
		if (cXsignificant <= 4)
			oxrunerror("Need 4 significant regressors");
		vpar[strfind(m_asPar_RM, "x1")]   =  8 / scale;
		vpar[strfind(m_asPar_RM, "x2")]   =  6 / scale;
		vpar[strfind(m_asPar_RM, "x3")]   =  4 / scale;
		vpar[strfind(m_asPar_RM, "x4")]   =  3 / scale;
		vpar[strfind(m_asPar_RM, "x1_1")] = -8 / scale;
		vpar[strfind(m_asPar_RM, "x2_1")] = -6 / scale;
		vpar[strfind(m_asPar_RM, "x3_1")] = -4 / scale;
		vpar[strfind(m_asPar_RM, "x4_1")] = -3 / scale;
	}
	
	if (cYlag && dYlag != 0)
		vpar[strfind(m_asPar_RM, "y_1")] = dYlag;
		
	decl dpar_constant = 0, dpar_trend = 0;
	if (sizerc(dTrend) == 2)
	{
		dpar_constant = dTrend[0];
		if (dTrend[1] != 0)
			dpar_trend = dTrend[1];
	}
	else
	{
		dpar_trend = dTrend;
	}

	if (dpar_constant != 0)
		vpar[strfind(m_asPar_RM, "Constant")] = dpar_constant / sqrt(GetObsCount());
	if (dpar_trend != 0 && !isnan(dpar_trend))
		vpar[strfind(m_asPar_RM, "Trend")] = dpar_trend / sqrt(GetObsCount());

	if (dFraction != 0)
		vpar = AddImpulseBreakToRefPar(vpar, dFraction, dSize,
			iDummySaturation == SAT_SIS || iDummySaturation == SAT_IIS_SIS);

	SetDGP(vpar);
}
///////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////
// Breaks and IIS
/**
Breaks and IIS Monte Carlo experiments.
@param dFraction
@param dSize size of the break
D3 has dFraction >= 2: break of size dSize in every dFraction'th observation
D2 has dFraction >= 0, <2: break of size dSize in first round(dFraction * T) observations
D1 has dFraction >= -1, <0: break of size dSize in last round(-dFraction * T) observations
D4 has dFraction >= -100, <-1: break of size dSize in 5 blocks of round(-dFraction/100 * T / 5) obs
evenly spread as: [bbbb .... bbbb .... bbbb .... bbbb .... bbbb]
*/
Breaks::Breaks(const obj, const iYvar, const iXvar, const iUvar, const dFraction, const dSize)
{
	JEDC_raw(obj, iYvar, iXvar, iUvar);

	SetGUMandDGP(0, 0, 0, 0, 0, USE_CONSTANT_U, 100, 0, TRUE,
		SAT_IIS, dFraction, dSize, 0, 0);

	decl sexp = "??";
	if (dFraction >= 2)
	{	// break of size dSize in every dFraction'th observation
		sexp = "D3";
	}
	else if (dFraction >= 0)
	{	// break of size dSize in first round(dFraction * T) observations
		sexp = "D2";
	}
	else if (dFraction >= -1)
	{	// break of size dSize in last round(-dFraction * T) observations
		sexp = "D1";
	}
	else if (dFraction >= -100)
	{	// break of size dSize in 5 blocks of round(-dFraction/100 * T / 5) obs
		// evenly spread as: [bbbb .... bbbb .... bbbb .... bbbb .... bbbb]
		sexp = "D4";
	}
	SetTitle(sprint(sexp, "(lambda=", dSize, ")\n"));
}
/**
Bc Monte Carlo experiments: break in last 20 observations.
@param dDeltaDGP delta: intercept in DGP
@param dSize size of the break
@param iConstantMode treatment of constant: 	USE_CONSTANT_NONE, USE_CONSTANT_U, USE_CONSTANT_X
*/
Bc::Bc(const obj, const iYvar, const iXvar, const iUvar, const dDeltaDGP, const dSize, const iConstantMode)
{
	JEDC_raw(obj, iYvar, iXvar, iUvar);
	decl ct = 100;
	SetGUMandDGP(0, 0, 0, 0, 0, iConstantMode, ct, 0, TRUE, SAT_IIS, -0.2, dSize, dDeltaDGP != 0 ? sqrt(ct)*dDeltaDGP|0 : 0, 0);
	SetTitle(sprint("Bc(delta=", dDeltaDGP, ", gamma=", dSize, ", ", GetRefConstantLabel(), ")\n"));
}
/**
B20 Monte Carlo experiments: break in every 5th observation.
@param dDeltaDGP delta (mean/10) in DGP
@param dSize size of the break
@param iConstantMode treatment of constant: 	USE_CONSTANT_NONE, USE_CONSTANT_U, USE_CONSTANT_X
*/
B20::B20(const obj, const iYvar, const iXvar, const iUvar, const dDeltaDGP, const dSize, const iConstantMode)
{
	JEDC_raw(obj, iYvar, iXvar, iUvar);
	SetGUMandDGP(0, 0, 0, 0, 0, iConstantMode, 100, 0, TRUE, SAT_IIS, 5, dSize, dDeltaDGP != 0 ? 10*dDeltaDGP|0 : 0, 0);
	SetTitle(sprint("B20(delta=", dDeltaDGP, ", gamma=", dSize, ", ", GetRefConstantLabel(), ")\n"));
}
/**
Bct Monte Carlo experiments: break in last 20 observations, both Constant and Trend
@param dDeltaDGP delta (mean/10) in DGP
@param dSize size of the break
@param iConstantMode treatment of constant: 	USE_CONSTANT_NONE, USE_CONSTANT_U, USE_CONSTANT_X
*/
Bct::Bct(const obj, const iYvar, const iXvar, const iUvar, const dDeltaDGP, const dSize, const iConstantMode)
{
	JEDC_raw(obj, iYvar, iXvar, iUvar);
	SetGUMandDGP(0, 0, 0, 0, 0, iConstantMode, 100, 0, TRUE, SAT_IIS, -0.2, dSize, 10*dDeltaDGP|0.2, 0);
	SetTitle(sprint("Bct(delta=", dDeltaDGP, ", gamma=", dSize, ", ", GetRefConstantLabel(), ")\n"));
}
/**											  
MBc Monte Carlo experiments: multiple breaks: 5 blocks of 4.
@param dDeltaDGP delta (mean/10) in DGP
@param dSize size of the break
@param iConstantMode treatment of constant: 	USE_CONSTANT_NONE, USE_CONSTANT_U, USE_CONSTANT_X
*/
MBc::MBc(const obj, const iYvar, const iXvar, const iUvar, const dDeltaDGP, const dSize, const iConstantMode)
{
	JEDC_raw(obj, iYvar, iXvar, iUvar);
	SetGUMandDGP(0, 0, 0, 0, 0, iConstantMode, 100, 0, TRUE, SAT_IIS, -20, dSize, dDeltaDGP != 0 ? 10*dDeltaDGP|0 : 0, 0);
	SetTitle(sprint("MBc(delta=", dDeltaDGP, ", gamma=", dSize, ", ", GetRefConstantLabel(), ")\n"));
}
/**											  
MBct Monte Carlo experiments: multiple breaks: 5 blocks of 4, both Constant and Trend.
@param dDeltaDGP delta (mean/10) in DGP
@param dSize size of the break
@param iConstantMode treatment of constant: 	USE_CONSTANT_NONE, USE_CONSTANT_U, USE_CONSTANT_X
*/
MBct::MBct(const obj, const iYvar, const iXvar, const iUvar, const dDeltaDGP, const dSize, const iConstantMode)
{
	JEDC_raw(obj, iYvar, iXvar, iUvar);
	SetGUMandDGP(0, 0, 0, 0, 0, iConstantMode, 100, 0, TRUE, SAT_IIS, -20, dSize, 10*dDeltaDGP|0.2, 0);
	SetTitle(sprint("MBct(delta=", dDeltaDGP, ", gamma=", dSize, ", ", GetRefConstantLabel(), ")\n"));
}
/**
BLc Monte Carlo experiments: break in last 20 observations with lagged dependent variable.
@param dDeltaDGP delta (mean/10) in DGP
@param dSize size of the break
@param iConstantMode treatment of constant: 	USE_CONSTANT_NONE, USE_CONSTANT_U, USE_CONSTANT_X
*/
BLc::BLc(const obj, const iYvar, const iXvar, const iUvar, const dDeltaDGP, const dSize, const iConstantMode)
{
	JEDC_raw(obj, iYvar, iXvar, iUvar);
	SetGUMandDGP(0, 1, 0, 0, 0, iConstantMode, 100, 0, TRUE, SAT_IIS, -0.2, dSize, dDeltaDGP != 0 ? 10*dDeltaDGP|0 : 0, 0.5);
	SetTitle(sprint("BLc(delta=", dDeltaDGP, ", gamma=", dSize, ", ", GetRefConstantLabel(), ")\n"));
}
/**
IUc Monte Carlo experiments: break in T-20th observations with unit-root lagged dependent variable.
@param dDeltaDGP delta (mean/10) in DGP
@param dSize size of the break
@param iConstantMode treatment of constant: 	USE_CONSTANT_NONE, USE_CONSTANT_U, USE_CONSTANT_X
*/
IUc::IUc(const obj, const iYvar, const iXvar, const iUvar, const dDeltaDGP, const dSize, const iConstantMode, const bUseModelDeltaY)
{
	decl ct = 100, sdgum = "";
	JEDC_raw(obj, iYvar, iXvar, iUvar);
	SetGUMandDGP(0, 1, 0, 0, 0, iConstantMode, ct, 0, TRUE, SAT_IIS, -0.2, dSize ~ zeros(1, ct * 0.2), 10*dDeltaDGP|.NaN, 1);
	UseModelDeltaY(bUseModelDeltaY);
	SetTitle(sprint("IUc(delta=", dDeltaDGP, ", gamma=", dSize, ", ", GetRefConstantLabel(), bUseModelDeltaY ? ", DGUM" : "", ")\n"));
}
/**
BUc Monte Carlo experiments: break in last 20 observations with unit-root lagged dependent variable.
@param dDeltaDGP delta (mean/10) in DGP
@param dSize size of the break
@param iConstantMode treatment of constant: 	USE_CONSTANT_NONE, USE_CONSTANT_U, USE_CONSTANT_X
*/
BUc::BUc(const obj, const iYvar, const iXvar, const iUvar, const dDeltaDGP, const dSize, const iConstantMode, const bUseModelDeltaY)
{
	JEDC_raw(obj, iYvar, iXvar, iUvar);
	SetGUMandDGP(0, 1, 0, 0, 0, iConstantMode, 100, 0, TRUE, SAT_IIS, -0.2, dSize, 10*dDeltaDGP|.NaN, 1);
	UseModelDeltaY(bUseModelDeltaY);
	SetTitle(sprint("BUc(delta=", dDeltaDGP, ", gamma=", dSize, ", ", GetRefConstantLabel(), bUseModelDeltaY ? ", DGUM" : "", ")\n"));
}
/**											  
MBct Monte Carlo experiments: multiple breaks: 5 blocks of 1, unit root.
@param dDeltaDGP delta (mean/10) in DGP
@param dSize size of the break
@param iConstantMode treatment of constant: 	USE_CONSTANT_NONE, USE_CONSTANT_U, USE_CONSTANT_X
*/
MIUc::MIUc(const obj, const iYvar, const iXvar, const iUvar, const dDeltaDGP, const dSize, const iConstantMode, const bUseModelDeltaY)
{
	decl ct = 100;
	decl vsize = dSize ~ zeros(1, round(ct * 0.2));
	vsize[4] = vsize[8] = vsize[12] = vsize[16] = vsize[0];
	
	JEDC_raw(obj, iYvar, iXvar, iUvar);
	SetGUMandDGP(0, 1, 0, 0, 0, iConstantMode, ct, 0, TRUE, SAT_IIS, -20, vsize, 10*dDeltaDGP|.NaN, 1);
	UseModelDeltaY(bUseModelDeltaY);
	SetTitle(sprint("MIUc(delta=", dDeltaDGP, ", gamma=", dSize, ", ", GetRefConstantLabel(), bUseModelDeltaY ? ", DGUM" : "", ")\n"));
}
/**											  
MBct Monte Carlo experiments: multiple breaks: 5 blocks of 4, unit root.
@param dDeltaDGP delta (mean/10) in DGP
@param dSize size of the break
@param iConstantMode treatment of constant: 	USE_CONSTANT_NONE, USE_CONSTANT_U, USE_CONSTANT_X
*/
MBUc::MBUc(const obj, const iYvar, const iXvar, const iUvar, const dDeltaDGP, const dSize, const iConstantMode, const bUseModelDeltaY)
{
	JEDC_raw(obj, iYvar, iXvar, iUvar);
	SetGUMandDGP(0, 1, 0, 0, 0, iConstantMode, 100, 0, TRUE, SAT_IIS, -20, dSize, 10*dDeltaDGP|.NaN, 1);
	UseModelDeltaY(bUseModelDeltaY);
	SetTitle(sprint("MBUc(delta=", dDeltaDGP, ", gamma=", dSize, ", ", GetRefConstantLabel(), bUseModelDeltaY ? ", DGUM" : "", ")\n"));
}
/**
BLcx Monte Carlo experiments: break in last 20 observations with lagged dependent variable and regressors.
@param dDeltaDGP delta (mean/10) in DGP
@param dSize size of the break
@param dTval t-value of relevant regressors
@param dRho autoregressive coefficient
@param iConstantMode treatment of constant: 	USE_CONSTANT_NONE, USE_CONSTANT_U, USE_CONSTANT_X
@param iSaturation saturation mode: SAT_NONE, SAT_IIS, SAT_SIS, SAT_IIS_SIS
*/
BLcx::BLcx(const obj, const iYvar, const iXvar, const iUvar, const dDeltaDGP, const dSize, const dTval, const dRho, const iConstantMode, const iSaturation)
{
	JEDCdyn(obj, iYvar, iXvar, iUvar);
	SetGUMandDGP(dTval, 1, 10, 1, 5, iConstantMode, 100, dRho, TRUE, iSaturation, -0.2, dSize, dDeltaDGP != 0 ? 10*dDeltaDGP|0 : 0, 0.5);
	SetTitle(sprint("BLcx(delta=", dDeltaDGP, ", gamma=", dSize, ", ", GetRefConstantLabel(), ", t=", dTval, ", rho=", dRho, ", sat=", GetSaturationLabel(), ")\n"));
}
/**
BUc Monte Carlo experiments: break in last 20 observations with unit-root lagged dependent variable and regressors.
@param dDeltaDGP delta (mean/10) in DGP
@param dSize size of the break
@param dTval t-value of relevant regressors
@param dRho autoregressive coefficient
@param iConstantMode treatment of constant: 	USE_CONSTANT_NONE, USE_CONSTANT_U, USE_CONSTANT_X
@param iSaturation saturation mode: SAT_NONE, SAT_IIS, SAT_SIS, SAT_IIS_SIS
*/
BUcx::BUcx(const obj, const iYvar, const iXvar, const iUvar, const dDeltaDGP, const dSize, const dTval, const dRho, const iConstantMode, const iSaturation)
{
	JEDCdyn(obj, iYvar, iXvar, iUvar);
	SetGUMandDGP(dTval, 1, 10, 1, 5, iConstantMode, 100, dRho, TRUE, iSaturation, -0.2, dSize, dDeltaDGP != 0 ? 10*dDeltaDGP|0 : 0, 1);
	SetTitle(sprint("BUcx(delta=", dDeltaDGP, ", gamma=", dSize, ", ", GetRefConstantLabel(), ", t=", dTval, ", rho=", dRho, ", sat=", GetSaturationLabel(), ")\n"));
}
/**
BUcx Monte Carlo experiments: break in last 20 observations with unit-root lagged dependent variable and regressors.
@param dDeltaDGP delta (mean/10) in DGP
@param dSize size of the break
@param dTval t-value of relevant regressors
@param dRho autoregressive coefficient
@param iConstantMode treatment of constant: 	USE_CONSTANT_NONE, USE_CONSTANT_U, USE_CONSTANT_X
@param iSaturation saturation mode: SAT_NONE, SAT_IIS, SAT_SIS, SAT_IIS_SIS
*/
BUcx_t::BUcx_t(const obj, const iYvar, const iXvar, const iUvar, const dDeltaDGP, const dSize, const dTval, const dRho, const iConstantMode, const iSaturation)
{
	JEDCdyn(obj, iYvar, iXvar, iUvar);
	SetGUMandDGP(dTval, 1, 10, 1, 5, iConstantMode, 100, dRho, TRUE, iSaturation, -0.2, dSize, 10*dDeltaDGP|.NaN, 1);
	SetTitle(sprint("BUcx_t(delta=", dDeltaDGP, ", gamma=", dSize, ", ", GetRefConstantLabel(), ", t=", dTval, ", rho=", dRho, ", sat=", GetSaturationLabel(), ")\n"));
}
///////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////
/**
Combines a range of simulation designs (all derived from the `SimDesign` class)
and a model estimation class (derived from Modelbase) together with
model selection methods into a Monte Carlo experiment.
<p>
Provides a way to set Autometrics options, although the derived class must decide
how this is passed on to the actual Autometrics object. For example, PcGive can
be used with Autometrics built-in, while PcFimlEx needs to be combined with an Autometrics
object. The latter sets Autometrics options directly, while the former uses the PcGive interface.
Here, the PcGive way is used.
*/
BaseExp::BaseExp()
{
	m_dAutoPval = 0.05;
	m_sAutoOutliers = "";
	m_bChopLags = TRUE;

	m_sMethod = "Autometrics";

	m_asAutoSet = m_aAutoSetVal = {};
}
/**
Selects a Monte Carlo experimental design.
@param oModel model object to hand to `SimDesign`
@param sDGP name of DGP
@param aDGPno number of DGP	variation, or array with DGP options
@param iYvar identifies endogenous variables
@param iXvar identifies free regressors
@param iUvar identifies unrerstricted (fixed) regressors
*/
BaseExp::SelectDesign(const oModel, const sDGP, const aDGPno, const iYvar, const iXvar, const iUvar)
{
	switch_single(sDGP)
	{
	case "HP":
		m_oDesign = new HP(aDGPno, oModel, iYvar, iXvar, iUvar);
	case "HPBig":
		m_oDesign = new HPBig(aDGPno, oModel, iYvar, iXvar, iUvar);
	case "JEDC":
		m_oDesign = new JEDC(oModel, iYvar, iXvar, iUvar, aDGPno);
	case "JEDC_EJ":
		m_oDesign = new JEDC_EJ(oModel, iYvar, iXvar, iUvar, aDGPno);
	case "D1":
		m_oDesign = new Breaks(oModel, iYvar, iXvar, iUvar, -0.2, aDGPno);
	case "D2":
		m_oDesign = new Breaks(oModel, iYvar, iXvar, iUvar, 0.2, aDGPno);
	case "D3":
		m_oDesign = new Breaks(oModel, iYvar, iXvar, iUvar, 5, aDGPno);
	case "Bc":
		m_oDesign = new Bc(oModel, iYvar, iXvar, iUvar, aDGPno[0], aDGPno[1], aDGPno[2]);
	case "B20":
		m_oDesign = new B20(oModel, iYvar, iXvar, iUvar, aDGPno[0], aDGPno[1], aDGPno[2]);
	case "Bct":
		m_oDesign = new Bct(oModel, iYvar, iXvar, iUvar, aDGPno[0], aDGPno[1], aDGPno[2]);
	case "MBc":
		m_oDesign = new MBc(oModel, iYvar, iXvar, iUvar, aDGPno[0], aDGPno[1], aDGPno[2]);
	case "MBct":
		m_oDesign = new MBct(oModel, iYvar, iXvar, iUvar, aDGPno[0], aDGPno[1], aDGPno[2]);
	case "BLc":
		m_oDesign = new BLc(oModel, iYvar, iXvar, iUvar, aDGPno[0], aDGPno[1], aDGPno[2]);
	case "IUc":
		m_oDesign = new IUc(oModel, iYvar, iXvar, iUvar, aDGPno[0], aDGPno[1], aDGPno[2], aDGPno[3]);
	case "BUc":
		m_oDesign = new BUc(oModel, iYvar, iXvar, iUvar, aDGPno[0], aDGPno[1], aDGPno[2], aDGPno[3]);
	case "MIUc":
		m_oDesign = new MIUc(oModel, iYvar, iXvar, iUvar, aDGPno[0], aDGPno[1], aDGPno[2], aDGPno[3]);
	case "MBUc":
		m_oDesign = new MBUc(oModel, iYvar, iXvar, iUvar, aDGPno[0], aDGPno[1], aDGPno[2], aDGPno[3]);
	case "BLcx":
		m_oDesign = new BLcx(oModel, iYvar, iXvar, iUvar, aDGPno[0], aDGPno[1], aDGPno[2], aDGPno[3], aDGPno[4], aDGPno[5]);
	case "BUcx":
		m_oDesign = new BUcx(oModel, iYvar, iXvar, iUvar, aDGPno[0], aDGPno[1], aDGPno[2], aDGPno[3], aDGPno[4], aDGPno[5]);
	case "BUcx_t":
		m_oDesign = new BUcx_t(oModel, iYvar, iXvar, iUvar, aDGPno[0], aDGPno[1], aDGPno[2], aDGPno[3], aDGPno[4], aDGPno[5]);
	default:
		oxrunerror("Unknown experiment");
	}
}
BaseExp::~BaseExp()
{
	delete m_oDesign;
	delete m_oStore;
}
/**
Creates the `SimStore` Monte Carlo storage.
@param cM No of replications
*/
BaseExp::CreateStore(const cM)
{
	m_oStore = new SimStore(m_oDesign.GetRefParDGP(), m_oDesign.GetRefParNames(), m_oDesign.GetSigmaInDGP(), cM);
}
/**
Short version of `SimStore::StoreResults` for Modelbase-derived estimation object
@param 	oModel Model object
@param 	iStore Storage type, one of: 	SimStore::STORE_REP, SimStore::STORE_DGP, SimStore::STORE_GUM, SimStore::STORE_RSC
@param 	iRep replication index
*/
BaseExp::StoreModel(const oModel, const iStore, const iRep)
{
//	decl asp, vp, vpse, vpv;
//	[asp, vp, vpse, vpv] = m_oDesign.SubstituteStepDummies(oModel.GetParNames(), oModel.GetPar(), oModel.GetStdErr(),
//		oModel.AutometricsSignificance(), oModel.GetSelStart(), oModel.GetSelEnd());
//		
//	println(">>", asp, vp, vpse, vpv);
//		
//	m_oStore.StoreResults(iStore, iRep, 0, vp, vpse, vpv, asp,
//		sqrt(oModel.GetOmega()), oModel.GetLogLik(), oModel.GetcT(), <>, <>);

//	println(meanc(oModel.GetW()) * oModel.GetPar(), sumc(oModel.GetW() .== 0) | sumc(oModel.GetW() .== 1), oModel.GetW(), oModel.GetY() - oModel.GetW() * oModel.GetPar() ~ oModel.GetResiduals());

	m_oStore.StoreResults(iStore, iRep, 0, oModel.GetPar(), oModel.GetStdErr(), oModel.AutometricsSignificance(),
		oModel.GetParNames(), sqrt(oModel.GetOmega()), oModel.GetLogLik(), oModel.GetcT(), <>, <>);
	m_oStore.StoreDiagnostics(iStore, iRep, oModel.AutometricsDiagnostics());
}
/**
Main Autometrics settings.
@param dPValue 	significance for model selection (1 for no selection)
@param sOutliers one of: "none","large","iis","sis","iis+sis","diis","diis+iis"
@param bChopLags TRUE: lag presearch; FALSE: no	lag presearch
*/
BaseExp::Autometrics(const dPValue, const sOutliers, const bChopLags)
{
	m_dAutoPval = dPValue;
	m_sAutoOutliers = sOutliers;
	m_bChopLags = bChopLags;
	m_sMethod = "Autometrics";
}
/**
Set Autometrics option.
@param sOpt 	option label
@param val option value
*/
BaseExp::AutometricsSet(const sOpt, const val)
{
	m_asAutoSet ~= sOpt;
	m_aAutoSetVal ~= val;
}
/**
Run Stepwise, see derived class.
*/
BaseExp::Stepwise(const sMethod, const dPValue, const maxK, const iCrit)
{
	m_sMethod = sMethod;
	m_dAutoPval = dPValue;
	m_maxK = maxK;
	m_iStopCrit = iCrit;
}
///////////////////////////////////////////////////////////////////////

