#include <oxstd.h>
#include <arma.h>

#include "simstore.oxh"
#import "simutils"


///////////////////////////////////////////////////////////////////////
// SimStore
/**
Creates storage for a Monte Carlo experiment
@param 	vPar True parameter vector
@param 	asPar names of parameters
@param 	dSigma True residual standard error
@param 	cM number of replications
@param 	cTforc number of forecasts (default is 0)
*/
SimStore::SimStore(const vPar, const asPar, const dSigma, const cM, const cTforc)
{
	m_asPar = asPar;
	m_vPar = vec(vPar);
	m_cParOrig = sizerc(m_vPar);
	m_dSigma = dSigma;
	m_cM = cM;

	m_mMC_vPar_REP = m_mMC_vParSE_REP = m_mMC_vParPval_REP = nans(m_cParOrig, cM);
	m_vMC_dESE_REP = m_vMC_dLogLik_REP = m_vMC_dSC_REP = m_vMC_cPar_REP = nans(1, cM);
	m_vMC_dWeight_REP = nans(1, cM);
	m_mMC_vForc_REP = nans(cTforc, cM);
	
	
	m_mMC_vPar_DGP = m_mMC_vParSE_DGP = m_mMC_vParPval_DGP = nans(m_cParOrig, cM);
	m_vMC_dESE_DGP = m_vMC_dLogLik_DGP = m_vMC_dSC_DGP = m_vMC_cPar_DGP = nans(1, cM);
	m_mMC_vForc_DGP = m_mMC_vAct_DGP = nans(cTforc, cM);
	
	m_mMC_vPar_GUM = m_mMC_vParSE_GUM = m_mMC_vParPval_GUM = nans(m_cParOrig, cM);
	m_vMC_dESE_GUM = m_vMC_dLogLik_GUM = m_vMC_dSC_GUM = m_vMC_cPar_GUM = nans(1, cM);
	m_mMC_vForc_GUM = nans(cTforc, cM);

	m_mMC_vPar_RSC = m_mMC_vParSE_RSC = m_mMC_vParPval_RSC = nans(m_cParOrig, cM);
	m_vMC_dESE_RSC = m_vMC_dLogLik_RSC = m_vMC_dSC_RSC = m_vMC_cPar_RSC = nans(1, cM);
	m_vMC_dWeight_RSC = nans(1, cM);
	m_vMC_vParCondWeight_RSC = zeros(m_cParOrig, cM);
	m_mMC_vForc_RSC = nans(cTforc, cM);

	m_mMC_vForcAvg_0 = m_mMC_vForcAvg_1 = m_mMC_vForcAvg_2 = nans(cTforc, cM);

	m_mMC_vEncomp = <>;
	m_mMC_vDiagnosticsGUM = m_mMC_vDiagnosticsREP = m_mMC_vDiagnosticsDGP = <>;
}
/**
Gets parameters from storage
@param 	iStore Storage type, one of: 	STORE_REP, STORE_DGP, STORE_GUM, STORE_RSC
@param 	iRep replication index
@param 	bUnconditional (only for STORE_RSC), TRUE: unconditional, FALSE: weighted conditionally on selection
*/
SimStore::GetStoredPar(const iStore, const iRep, const bUnconditional)
{
	switch_single (iStore)
	{
		case STORE_DGP:
			return m_mMC_vPar_DGP[][iRep];
		case STORE_GUM:
			return m_mMC_vPar_GUM[][iRep];
		case STORE_REP:
			return m_mMC_vPar_REP[][iRep] / m_vMC_dWeight_REP[0][iRep];
		case STORE_RSC:
			if (bUnconditional)
				return m_mMC_vPar_RSC[][iRep] / m_vMC_dWeight_RSC[0][iRep];
			else
			{
				decl invw = m_vMC_vParCondWeight_RSC[][iRep];
				invw = invw .== 0 .? 0 .: 1 ./ invw;
				return m_mMC_vPar_RSC[][iRep] .* invw;
			}
	}
}
/**
Stores forecast from averaging
@param 	iStore Storage type, one of: 	STORE_FORC_AVG0, STORE_FORC_AVG1, STORE_FORC_AVG2
@param 	iRep replication index
@param 	vAvgForc forecasts
*/
SimStore::StoreForecastsFromAverage(const iStore, const iRep, const vAvgForc)
{
	switch_single (iStore)
	{
		case STORE_FORC_AVG0:
			m_mMC_vForcAvg_0[][iRep] = vAvgForc;
		case STORE_FORC_AVG1:
			m_mMC_vForcAvg_1[][iRep] = vAvgForc;
		case STORE_FORC_AVG2:
			m_mMC_vForcAvg_2[][iRep] = vAvgForc;
	}
}
/**
Stores results from a Monte Carlo replication
@param 	iStore Storage type, one of: 	STORE_REP (replication), STORE_DGP (estimated DGP),
	STORE_GUM (estimated GUM),
	STORE_RSC (exp(-SC/2) weighted replication, for forecast averaging)
@param 	iRep replication index
@param 	iRepStage 0: results stored, >=1: results added to previous results,
so keeping only sum of results (addition can only be used with STORE_REP and STORE_RSC)
@param 	vPar parameter vector
@param 	vParSE standard errors of parameters
@param 	vParPval pvalues of parameters
@param 	asPar names of parameters
@param 	dESE equation standard error
@param 	dLogLik log-likelihood
@param 	cT sample size
@param 	vForc forecasts (or <>)
@param 	vAct actual values for forecast period (or <>), for STORE_DGP only
*/
SimStore::StoreResults(const iStore, const iRep, const iRepStage, const vPar,
	const vParSE, const vParPval, const asPar, const dESE, const dLogLik,
	const cT, const vForc, const vAct)
{
	decl vpar, vparse, vparpval, vic, cp, idx;

	cp = sizeof(asPar);
	// lookup of the supplied parameters in the reference set
	idx = strfind(m_asPar, asPar);

	if (any(idx .== -1))
	{
		serial
		{
			// it could be that variables are added to GUM later,e.g. outliers
			decl idxnew = vecindex(idx .== -1), cnew = sizerc(idxnew);
			m_asPar ~= array(asPar[idxnew]);
			m_vPar |= zeros(cnew, 1);
	
			m_mMC_vPar_DGP |= zeros(cnew, sizec(m_mMC_vPar_DGP));
			m_mMC_vParSE_DGP |= zeros(cnew, sizec(m_mMC_vParSE_DGP));
			m_mMC_vParPval_DGP |= constant(.NaN, cnew, sizec(m_mMC_vParPval_DGP));
			m_mMC_vPar_GUM |= zeros(cnew, sizec(m_mMC_vPar_GUM));
			m_mMC_vParSE_GUM |= zeros(cnew, sizec(m_mMC_vParSE_GUM));
			m_mMC_vParPval_GUM |= constant(.NaN, cnew, sizec(m_mMC_vParPval_GUM));
			m_mMC_vPar_REP |= zeros(cnew, sizec(m_mMC_vPar_REP));
			m_mMC_vParSE_REP |= zeros(cnew, sizec(m_mMC_vParSE_REP));
			m_mMC_vParPval_REP |= constant(.NaN, cnew, sizec(m_mMC_vParPval_REP));
			m_mMC_vPar_RSC |= zeros(cnew, sizec(m_mMC_vPar_RSC));
			m_mMC_vParSE_RSC |= zeros(cnew, sizec(m_mMC_vParSE_RSC));
			m_mMC_vParPval_RSC |= constant(.NaN, cnew, sizec(m_mMC_vParPval_RSC));
	
			println("Warning, added: ", asPar[idxnew], " (ignored for size)");
	
			idx = strfind(m_asPar, asPar);
		}
	}
	vpar = zeros(sizeof(m_asPar), 1);
	vpar[idx][] = vPar;
	
	vparse = zeros(sizeof(m_asPar), 1);
	vparse[idx][] = vParSE;

	vparpval = constant(.NaN, sizeof(m_asPar), 1);
	vparpval[idx][] = vParPval;

	vic = InfoCrit(dLogLik, cp, cT);
	
	switch_single (iStore)
	{
		case STORE_DGP:
			m_mMC_vPar_DGP[][iRep] = vpar;
			m_mMC_vParSE_DGP[][iRep] = vparse;
			m_mMC_vParPval_DGP[][iRep] = vparpval;
			m_vMC_dESE_DGP[][iRep] = dESE;
			m_vMC_dLogLik_DGP[][iRep] = dLogLik;
			m_vMC_dSC_DGP[][iRep] = vic[2];
			m_vMC_cPar_DGP[][iRep] = cp;
			m_mMC_vForc_DGP[][iRep] = vForc;
			m_mMC_vAct_DGP[][iRep] = vAct;
		case STORE_GUM:
			m_mMC_vPar_GUM[][iRep] = vpar;
			m_mMC_vParSE_GUM[][iRep] = vparse;
			m_mMC_vParPval_GUM[][iRep] = vparpval;
			m_vMC_dESE_GUM[][iRep] = dESE;
			m_vMC_dLogLik_GUM[][iRep] = dLogLik;
			m_vMC_dSC_GUM[][iRep] = vic[2];
			m_vMC_cPar_GUM[][iRep] = cp;
			m_mMC_vForc_GUM[][iRep] = vForc;
		case STORE_REP:
			if (iRepStage == 0)
			{
				m_mMC_vPar_REP[][iRep] = vpar;
				m_mMC_vParSE_REP[][iRep] = vparse;
				m_mMC_vParPval_REP[][iRep] = vparpval;
				m_vMC_dESE_REP[][iRep] = dESE;
				m_vMC_dLogLik_REP[][iRep] = dLogLik;
				m_vMC_dSC_REP[][iRep] = vic[2];
				m_vMC_cPar_REP[][iRep] = cp;
				m_mMC_vForc_REP[][iRep] = vForc;
				m_vMC_dWeight_REP[][iRep] = 1;
			}
			else
			{
				serial
				{
					m_mMC_vPar_REP[][iRep] += vpar;
					m_mMC_vParSE_REP[][iRep] += vparse;
					m_mMC_vParPval_REP[][iRep] += vparpval;
					m_vMC_dESE_REP[][iRep] += dESE;
					m_vMC_dLogLik_REP[][iRep] += dLogLik;
					m_vMC_dSC_REP[][iRep] += vic[2];
					m_vMC_cPar_REP[][iRep] += cp;
					if (sizerc(vForc))
						m_mMC_vForc_REP[][iRep] += vForc;
					m_vMC_dWeight_REP[][iRep] += 1;
				}
			}
		case STORE_RSC:
		{
			decl w = exp(-vic[2] / 2);
			decl wpar_cond = w .* (vpar .!= 0);
			
			if (iRepStage == 0)
			{
				m_mMC_vPar_RSC[][iRep] = vpar * w;
				m_mMC_vParSE_RSC[][iRep] = vparse * w;
				m_mMC_vParPval_RSC[][iRep] = vparpval * w;
				m_vMC_dESE_RSC[][iRep] = dESE * w;
				m_vMC_dLogLik_RSC[][iRep] = dLogLik * w;
				m_vMC_dSC_RSC[][iRep] = vic[2] * w;
				m_vMC_cPar_RSC[][iRep] = cp * w;
				m_mMC_vForc_RSC[][iRep] = vForc * w;
				m_vMC_dWeight_RSC[][iRep] = w;
				m_vMC_vParCondWeight_RSC[][iRep] = wpar_cond;
			}
			else
			{
				serial
				{
					// NB: always do first with iRepStage==0 to initialize
					m_mMC_vPar_RSC[][iRep] += vpar * w;
					m_mMC_vParSE_RSC[][iRep] += vparse * w;
					m_mMC_vParPval_RSC[][iRep] += vparpval * w;
					m_vMC_dESE_RSC[][iRep] += dESE * w;
					m_vMC_dLogLik_RSC[][iRep] += dLogLik * w;
					m_vMC_dSC_RSC[][iRep] += vic[2] * w;
					m_vMC_cPar_RSC[][iRep] += cp * w;
					if (sizerc(vForc))
						m_mMC_vForc_RSC[][iRep] += vForc * w;
					m_vMC_dWeight_RSC[][iRep] += w;
					m_vMC_vParCondWeight_RSC[][iRep] += wpar_cond;
				}
			}
		}
	}
}
/**
Add encompassing results to storage
@param 	iRep replication index
@param 	vEnc encompassing test(s)
*/
SimStore::StoreEncompassing(const iRep, const vEnc)
{
	if (sizerc(m_mMC_vEncomp) == 0)
	{
		serial
		{
			m_mMC_vEncomp = nans(sizerc(vEnc), m_cM);
		}
	}
	m_mMC_vEncomp[][iRep] = vec(vEnc);
}
/**
Add diagnostic results to storage
@param 	iStore Storage type, one of: 	STORE_REP (replication), STORE_DGP (estimated DGP),
	STORE_GUM (estimated GUM),
	STORE_RSC (exp(-SC/2) weighted replication, for forecast averaging)
@param 	iRep replication index
*/
SimStore::StoreDiagnostics(const iStore, const iRep, const vDiag)
{
	switch_single (iStore)
	{
		case STORE_DGP:
			if (sizerc(m_mMC_vDiagnosticsDGP) == 0)
			{
				serial { m_mMC_vDiagnosticsDGP = nans(sizerc(vDiag), m_cM); }
			}
			m_mMC_vDiagnosticsDGP[][iRep] = vec(vDiag);
		case STORE_GUM:
			if (sizerc(m_mMC_vDiagnosticsGUM) == 0)
			{
				serial { m_mMC_vDiagnosticsGUM = nans(sizerc(vDiag), m_cM); }
			}
			m_mMC_vDiagnosticsGUM[][iRep] = vec(vDiag);
		case STORE_REP:
			if (sizerc(m_mMC_vDiagnosticsREP) == 0)
			{
				serial { m_mMC_vDiagnosticsREP = nans(sizerc(vDiag), m_cM); }
			}
			m_mMC_vDiagnosticsREP[][iRep] = vec(vDiag);
	}

	
}
/**
Report diagnostic results
@param asDiag Names of diagnostic tests
*/
SimStore::ReportDiagnostics(const asDiag)
{
	if (sizerc(m_mMC_vDiagnosticsDGP))
		println("\nDiagnostic tests DGP rejection(%)",
			"%c", {"p=0.10","p=0.05","p=0.01"}, "%7.2f",
			"%r", asDiag ~ {"Joint"},
			100 * (meanr(m_mMC_vDiagnosticsDGP .< 0.1) ~ meanr(m_mMC_vDiagnosticsDGP .< 0.05) ~ meanr(m_mMC_vDiagnosticsDGP .< 0.01))
		  	| 100 * (meanr(sumc(m_mMC_vDiagnosticsDGP .< 0.1) .!= 0) ~ meanr(sumc(m_mMC_vDiagnosticsDGP .< 0.05) .!= 0) ~ meanr(sumc(m_mMC_vDiagnosticsDGP .< 0.01) .!= 0)));
	if (sizerc(m_mMC_vDiagnosticsREP))
		println("Diagnostic tests Model rejection(%)",
			"%c", {"p=0.10","p=0.05","p=0.01"}, "%7.2f",
			"%r", asDiag ~ {"Joint"},
			100 * (meanr(m_mMC_vDiagnosticsREP .< 0.1) ~ meanr(m_mMC_vDiagnosticsREP .< 0.05) ~ meanr(m_mMC_vDiagnosticsREP .< 0.01))
		  	| 100 * (meanr(sumc(m_mMC_vDiagnosticsREP .< 0.1) .!= 0) ~ meanr(sumc(m_mMC_vDiagnosticsREP .< 0.05) .!= 0) ~ meanr(sumc(m_mMC_vDiagnosticsREP .< 0.01) .!= 0)));
	if (sizerc(m_mMC_vDiagnosticsGUM))
		println("Diagnostic tests GUM rejection(%)",
			"%c", {"p=0.10","p=0.05","p=0.01"}, "%7.2f",
			"%r", asDiag ~ {"Joint"},
			100 * (meanr(m_mMC_vDiagnosticsGUM .< 0.1) ~ meanr(m_mMC_vDiagnosticsGUM .< 0.05) ~ meanr(m_mMC_vDiagnosticsGUM .< 0.01))
		  	| 100 * (meanr(sumc(m_mMC_vDiagnosticsGUM .< 0.1) .!= 0) ~ meanr(sumc(m_mMC_vDiagnosticsGUM .< 0.05) .!= 0) ~ meanr(sumc(m_mMC_vDiagnosticsGUM .< 0.01) .!= 0)));
}

/**
Report parameter estimation results
@param asColumn Names of final columns (or {})
{}: no additional names, last column is called final (STORE_REP)
{"x"}: one additional name, last column is called x (STORE_REP)
{"x","y"}: two additional names, last columns are (STORE_REP, STORE_RSC, STORE_RSC conditional)
@param asColumn Names of forecast columns (or {})
*/
SimStore::ReportPar(asColumn, asColumnForc)
{
	decl vpar_rep = m_mMC_vPar_REP ./ m_vMC_dWeight_REP;
	decl vpar_rsc = m_mMC_vPar_RSC ./ m_vMC_dWeight_RSC;
	decl vpar_rsc_cond = m_mMC_vPar_RSC ./ m_vMC_vParCondWeight_RSC;
	decl vparse_rep = m_mMC_vParSE_REP ./ m_vMC_dWeight_REP;
	decl vparse_rsc = m_mMC_vParSE_RSC ./ m_vMC_dWeight_RSC;
	decl vforc_rep = m_mMC_vForc_REP ./ m_vMC_dWeight_REP;
	decl vforc_rsc = m_mMC_vForc_RSC ./ m_vMC_dWeight_RSC;
	decl vpar_gum = m_mMC_vPar_GUM, vparse_gum = m_mMC_vParSE_GUM;

	if (sizerc(vpar_gum) == 0)
		vpar_gum = vparse_gum = constant(.NaN, sizer(m_vPar), sizec(m_vPar));

	decl mpar = m_vPar ~ meanr(m_mMC_vPar_DGP) ~ meanr(vpar_gum) ~ meanr(vpar_rep);

	if (sizeof(asColumn) <= 1)
		vpar_rsc = vparse_rsc = <>;
	if (sizerc(vpar_rsc))
		mpar ~= meanr(vpar_rsc);
	if (sizeof(asColumn) > 2)
		mpar ~= sumr(isdotnan(vpar_rsc_cond) .? 0 .: vpar_rsc_cond) ./ sumr(!isdotnan(vpar_rsc_cond));

	if (sizeof(asColumn) == 0)
		asColumn = {"Final"};
	print("\nParameter Estimates (with zeros)");
	println("%r", m_asPar, "%c", {"True","DGP","GUM"} ~ asColumn, mpar);

	mpar = m_dSigma ~ meanr(m_vMC_dESE_DGP) ~
		(sizerc(m_vMC_dESE_GUM) ? meanr(m_vMC_dESE_GUM) : .NaN) ~ meanr(m_vMC_dESE_REP);
	mpar |= sqr(m_dSigma) ~ meanr(sqr(m_vMC_dESE_DGP)) ~
		(sizerc(m_vMC_dESE_GUM) ? meanr(sqr(m_vMC_dESE_GUM)) : .NaN) ~ meanr(sqr(m_vMC_dESE_REP));
	println("%r", {"sigma","sigma^2"}, "%c", {"True","DGP","GUM","Final"}, mpar);

	mpar = meanr(sqr(m_mMC_vPar_DGP - m_vPar)) ~ meanr(sqr(vpar_gum - m_vPar))
		~ meanr(sqr(vpar_rep - m_vPar)) ~ (sizerc(vpar_rsc) ? meanr(sqr(vpar_rsc - m_vPar)) : <>);
	print("\nMean Squared Parameter Errors (with zeros)");
	if (sizec(mpar) > 3)
		println("%r", m_asPar, "%c", {"MSE:DGP","MSE:GUM","MSE:Final"} ~ asColumn, mpar);
	else
		println("%r", m_asPar, "%c", {"MSE:DGP","MSE:GUM","MSE:Final","RMSE:DGP","RMSE:GUM","RMSE:Final"} ~ asColumn, mpar ~ sqrt(mpar));

	mpar =  meanr(sqr(m_vMC_dESE_DGP - m_dSigma)) ~
		(sizerc(m_vMC_dESE_GUM) ? meanr(sqr(m_vMC_dESE_GUM - m_dSigma)) : .NaN) ~
		meanr(sqr(m_vMC_dESE_REP - m_dSigma));
	mpar |=  meanr(sqr(sqr(m_vMC_dESE_DGP) - sqr(m_dSigma))) ~
		(sizerc(m_vMC_dESE_GUM) ? meanr(sqr(sqr(m_vMC_dESE_GUM) - sqr(m_dSigma))) : .NaN) ~
		meanr(sqr(sqr(m_vMC_dESE_REP) - sqr(m_dSigma)));
	println("%r", {"sigma","sigma^2"}, "%c", {"MSE:DGP","MSE:GUM","MSE:Final","RMSE:DGP","RMSE:GUM","RMSE:Final"}, mpar ~ sqrt(mpar));

	print("\nParameter Estimates (only when included)");
	println("%r", m_asPar, "%c", {"True","DGP","GUM"} ~ asColumn,
		m_vPar ~ meanr(m_mMC_vPar_DGP) ~ meanr(vpar_gum)
		~ sumr(vpar_rep) ./ sumr(vpar_rep .!= 0) ~ (sizerc(vpar_rsc) ? sumr(vpar_rsc) ./ sumr(vpar_rsc .!= 0) : <>));

	print("\nParameter Standard Errors (only when included)");
	println("%r", m_asPar, "%c", {"DGP","GUM"} ~ asColumn,
		meanr(m_mMC_vParSE_DGP)	~ meanr(vparse_gum)
		~ sumr(vparse_rep) ./ sumr(vparse_rep .!= 0)
		~ (sizerc(vpar_rsc) ? sumr(vparse_rsc) ./ sumr(vparse_rsc .!= 0) : <>));

	if (sizerc(vforc_rep) == 0)
		return {<>,{}};
	
	if (sizeof(asColumnForc) == 0)
		asColumnForc = {"Final"};
	decl as_msfe = {"DGP","GUM"} ~ asColumnForc;
	decl msfe = meanr(sqr(m_mMC_vForc_DGP - m_mMC_vAct_DGP))
		~ (sizerc(m_mMC_vForc_GUM) ? meanr(sqr(m_mMC_vForc_GUM - m_mMC_vAct_DGP)) : .NaN)
		~ meanr(sqr(vforc_rep - m_mMC_vAct_DGP))
		~ (sizerc(vforc_rsc) ? meanr(sqr(vforc_rsc - m_mMC_vAct_DGP)) : <>);

	if (!isnan(m_mMC_vForcAvg_0))
		msfe ~= meanr(sqr(m_mMC_vForcAvg_0 - m_mMC_vAct_DGP));
	if (!isnan(m_mMC_vForcAvg_1))
		msfe ~= meanr(sqr(m_mMC_vForcAvg_1 - m_mMC_vAct_DGP));		
	if (!isnan(m_mMC_vForcAvg_2))	
		msfe ~= meanr(sqr(m_mMC_vForcAvg_2 - m_mMC_vAct_DGP));

	if (sizerc(msfe))
	{
		print("\nMean Squared Forecast Errors");
		decl ash = new array[sizer(msfe)];
		for (decl i = 0; i < sizer(msfe); ++i)
			ash[i] = sprint("h=", i + 1);
		println("%c", as_msfe, "%r", ash, msfe);
	}

	return {msfe, as_msfe};
}
/**
Report summary results from selection
@param sTitle title
@param cParU number of unrestricted coefficients (to be excluded)
@param cT estimation sample size
@param dPVal1 p-value of selection (used to decide on significant/insignificant count; not used for gauge/potency)
@param dDominatingPVal p-value of for encompassing
*/
SimStore::ReportSelection(const sTitle, const cParU, const cT, const dPVal1, const dDominatingPVal)
{
	decl cm = sizec(m_mMC_vPar_REP);
	decl cp = m_cParOrig - cParU;
	// index of DGP variables
	decl gumidx_dgp = cp ? vecindex(m_vPar[ : cp - 1] .!= 0) : <>;
	decl gumidx_out = cp ? vecindex(m_vPar[ : cp - 1] .== 0) : <>;
	decl csig = sizerc(gumidx_dgp), cinsig = cp - csig;
	
	// hits: count of included DGP variables over replications
	decl hits = sumr(m_mMC_vPar_REP[gumidx_dgp][] .!= 0);
	
	// missing: for each replication: #DGP vars missing in final model;	0: DGP is nested in final 
	decl missing = sumc(m_mMC_vPar_REP[gumidx_dgp][] .== 0);
	// included: for each replication: #DGP vars included in final model 
	decl included = sumc(m_mMC_vPar_REP[gumidx_dgp][] .!= 0);
	// excess: for each replication: #non-DGP variables in the final model
	decl excess = sumc(m_mMC_vPar_REP[gumidx_out][] .!= 0);


	decl dtrue = sumc(hits) / cm;
	decl dfalse = sumr(sumc(m_mMC_vParPval_REP[gumidx_out][] .!= .NaN)) / cm;
	decl dfalse_signif = sumr(sumc(m_mMC_vParPval_REP[gumidx_out][] .!= .NaN .&& m_mMC_vParPval_REP[gumidx_out][] .<= dPVal1)) / cm;
	decl dfalse_insig = sumr(sumc(m_mMC_vParPval_REP[gumidx_out][] .!= .NaN .&& m_mMC_vParPval_REP[gumidx_out][] .> dPVal1)) / cm;

	decl ret_val =
			100 * dfalse / cinsig					// Gauge
			| 100 * dtrue / csig					// Potency
			| cm | csig | cinsig | m_cParOrig
			| dtrue									// True variables
			| sumr(sumc(m_mMC_vParPval_REP .!= .NaN .&& m_mMC_vParPval_REP .> dPVal1)) / cm	  // Insig
			| dfalse_insig							// Falsely Insignif
			| dfalse_signif							// Falsely signif
			| double(meanr(m_vMC_dESE_REP))
			| double(meanr(m_vMC_dESE_DGP))
			| meanr(missing .== 0 .&& excess .== 0)	* 100
			| meanr(missing .== 0) * 100
			| 100 * dfalse_signif / cinsig;				// Gauge (significant only)

	if (rows(m_mMC_vEncomp))
	{
		ret_val |=
			  100 * double(meanr(m_mMC_vEncomp[0][] .< dDominatingPVal))
			| 100 * double(meanr(m_mMC_vEncomp[1][] .< dDominatingPVal))
			| 100 * double(meanr(m_mMC_vEncomp[0][] .< dDominatingPVal .&& m_mMC_vEncomp[1][] .< dDominatingPVal));
	}
	
	println("\n*** ", sTitle, " M=", cm, " T=", cT, " pvalue=", dPVal1, " #insig=", cinsig, " #sig=", csig, " cp=", sizerc(m_vPar), " cp_orig=", m_cParOrig);

	decl sat = sprint(" at ", 100*dDominatingPVal, "%");
	println("%r", {
		"Gauge (%)","Potency (%)",
		"Replications, M=", "GUM: relevant variables", "GUM: irrelevant variables", "GUM: no of variables",
		"Relevant (avg no)","Insignificant (avg no)","Insign. irrelevant (avg no)","Signif. irrelevant (avg no)",
		"ESE final model", "ESE DGP",
		"Final equals DGP (% reps)",
		"Final contains DGP (% reps)",
		"Gauge (significant only)",
		"F+DGP dominates F" ~ sat,"F+DGP dominates DGP" ~ sat,"Both dominated" ~ sat},
		"%10.4f", ret_val);

	if (sizerc(hits))
		println("%r", {"Hits"}, "%c", array(m_asPar[gumidx_dgp]), "%7.1f", hits' * 100 / cm);
	else
		println("No Hits: DGP empty");

	println("Model size", "%5.0f", range(0,20) | countr(m_vMC_cPar_REP, range(0,19)));

	return ret_val;
}

///////////////////////////////////////////////////////////////////////

